/*
 * Decompiled with CFR 0.152.
 */
package kd.bos.gptas.adapter.vectordb.vdb.service;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicBoolean;
import kd.ai.vdb.Filter;
import kd.ai.vdb.QueryParameters;
import kd.ai.vdb.objects.CollectionData;
import kd.ai.vdb.objects.CollectionMeta;
import kd.ai.vdb.objects.DataType;
import kd.ai.vdb.objects.FieldMeta;
import kd.ai.vdb.objects.IdFieldMeta;
import kd.ai.vdb.objects.TokenizeType;
import kd.ai.vdb.objects.VectorFieldMeta;
import kd.ai.vdb.objects.VectorSimilarity;
import kd.bos.context.RequestContext;
import kd.bos.gptas.adapter.vectordb.vdb.client.VDBClientWrapper;
import kd.bos.gptas.adapter.vectordb.vdb.service.VDBServiceConverter;
import kd.bos.gptas.api.km.split.SplitConfig;
import kd.bos.gptas.api.vector.VectorResult;
import kd.bos.gptas.common.GptasErrorCode;
import kd.bos.gptas.common.VectorStorePlugin;
import kd.bos.gptas.common.vectordb.exception.VectorException;
import kd.bos.gptas.common.vectordb.model.VectorChunk;
import kd.bos.gptas.common.vectordb.model.VectorQuery;
import kd.bos.gptas.common.vectordb.service.KnowledgeVectorStoreService;
import kd.bos.logging.Log;
import kd.bos.logging.LogFactory;
import kd.bos.util.Pair;

class VDBService
extends KnowledgeVectorStoreService {
    private static final Log logger = LogFactory.getLog(VDBService.class);
    private static final int BATCH_SIZE = 1000;
    protected static final String VECTOR_FIELD_NAME = "vector";
    protected static final String[] OUT_FIELDS = new String[]{"repoId", "knowledgeId", "groupId"};
    protected static final String COLLECTION_NAME = "collectionName";
    private final AtomicBoolean initialized = new AtomicBoolean();
    protected final VDBClientWrapper vdbClient;
    protected final VDBServiceConverter converter;
    private final String formId;
    private List<VectorStorePlugin> vectorStorePluginList = new ArrayList<VectorStorePlugin>(16);
    private CollectionMeta collectionMeta;
    protected String collectionName;

    public VDBService(VDBClientWrapper vdbClient, String formId) {
        this.vdbClient = vdbClient;
        this.formId = formId;
        this.converter = new VDBServiceConverter(formId);
        this.collectionName = this.generateCollectionName();
        this.collectionMeta = this.buildCollectionMeta(this.collectionName);
    }

    public void setVectorStorePlugin(List<VectorStorePlugin> vectorStorePluginList) {
        this.vectorStorePluginList = vectorStorePluginList;
    }

    protected String getCollectionName() {
        return this.collectionName;
    }

    protected String getIDPrefix() {
        return "";
    }

    protected boolean enableVector(String entityId) {
        boolean enable = false;
        for (VectorStorePlugin vectorStorePlugin : this.vectorStorePluginList) {
            SplitConfig splitConfig = vectorStorePlugin.getSplitConfig(entityId);
            enable = splitConfig.isEnableVector();
        }
        return enable;
    }

    protected boolean enableFullText(String entityId) {
        boolean enable = false;
        for (VectorStorePlugin vectorStorePlugin : this.vectorStorePluginList) {
            SplitConfig splitConfig = vectorStorePlugin.getSplitConfig(entityId);
            enable = splitConfig.isEnableFullText();
        }
        return enable;
    }

    public synchronized void initialize() throws VectorException {
        if (this.initialized.compareAndSet(false, true)) {
            try {
                logger.info("Initializing vector database collection: {}", (Object)this.collectionName);
                this.vdbClient.createCollection(this.collectionMeta);
                logger.info("Successfully initialized vector database collection: {}", (Object)this.collectionName);
            }
            catch (VectorException e) {
                this.initialized.set(false);
                throw e;
            }
        }
    }

    protected CollectionMeta buildCollectionMeta(String collectionName) {
        CollectionMeta collection = new CollectionMeta();
        collection.setName(collectionName);
        collection.addField((FieldMeta)new IdFieldMeta());
        collection.addField((FieldMeta)new VectorFieldMeta(VECTOR_FIELD_NAME, DataType.DOUBLE, this.vdbClient.getDimension(), VectorSimilarity.COSINE));
        collection.addField(new FieldMeta("repoId", DataType.STRING));
        collection.addField(new FieldMeta("knowledgeId", DataType.INT64));
        collection.addField(new FieldMeta("groupId", DataType.INT64));
        collection.addField((FieldMeta)FieldMeta.text((String)"title", (TokenizeType)TokenizeType.DEFAULT));
        return collection;
    }

    public List<VectorResult> doSearch(VectorQuery query) throws VectorException {
        this.validateQuery(query);
        Long l = System.currentTimeMillis();
        logger.debug("Performing vector search: collection={}, topK={}", (Object)this.collectionName, (Object)Objects.toString(query.getTopK(), "NULL"));
        try {
            if (!this.initialized.get()) {
                this.initialize();
            }
            List queryVector = query.getQueryVector();
            double[] destVector = new double[queryVector.size()];
            for (int i = 0; i < queryVector.size(); ++i) {
                destVector[i] = ((Float)queryVector.get(i)).doubleValue();
            }
            Filter filter = this.buildSearchFilter(query);
            ArrayList<VectorResult> vectorResults = new ArrayList<VectorResult>(16);
            boolean enableVector = this.enableVector(this.formId);
            logger.info("vector search set: {} ", (Object)enableVector);
            if (enableVector) {
                vectorResults.addAll(this.doSearch(query, destVector, filter));
            }
            boolean enableFullText = this.enableFullText(this.formId);
            logger.info("fulltext search set: {} ", (Object)enableFullText);
            if (enableFullText) {
                vectorResults.addAll(this.doSearchFullText(query, filter));
            }
            List<VectorResult> filteredResults = this.filterResults(vectorResults, Float.valueOf(query.getLimitScore()));
            logger.info("Search completed: total results={}, filtered results={},cost={} ms.", new Object[]{vectorResults != null ? vectorResults.size() : 0, filteredResults != null ? filteredResults.size() : 0, System.currentTimeMillis() - l});
            return filteredResults;
        }
        catch (Exception e) {
            logger.error("Vector search operation failed", (Throwable)e);
            throw new VectorException(GptasErrorCode.VECTOR_STORE_READ_ERROR, "Vector search failed: " + this.getErrorMessage(e), (Throwable)e);
        }
    }

    protected List<VectorResult> doSearchFullText(VectorQuery query, Filter filter) throws VectorException {
        QueryParameters params = new QueryParameters(this.getCollectionName(), query.getTopK(), OUT_FIELDS);
        Filter fullTextFilter = Filter.textMatch((String)"title", (String)query.getQueryText(), (TokenizeType)TokenizeType.DEFAULT, (boolean)true, (float)1.0f);
        Filter searchFilter = Filter.and1((Filter[])new Filter[]{filter, fullTextFilter});
        params.setFilter(searchFilter);
        CollectionData collectionData = this.vdbClient.query(params);
        List<VectorResult> vectorResults = this.converter.toVectorResults(collectionData, this.getIDPrefix());
        vectorResults.forEach(o -> o.setScoreFrom("fulltext"));
        return vectorResults;
    }

    private String getErrorMessage(Throwable e) {
        StringBuilder sb = new StringBuilder(16);
        while (e != null) {
            sb.append(e.getMessage()).append("; ");
            e = e.getCause();
        }
        return sb.toString();
    }

    protected List<VectorResult> doSearch(VectorQuery query, double[] destVector, Filter filter) {
        CollectionData collectionData = this.vdbClient.search(this.getCollectionName(), VECTOR_FIELD_NAME, destVector, query.getTopK(), OUT_FIELDS, filter);
        return this.converter.toVectorResults(collectionData, this.getIDPrefix());
    }

    private Filter buildSearchFilter(VectorQuery query) {
        List<Filter> conditions = this.getFilterList(query);
        if (conditions.isEmpty()) {
            return null;
        }
        if (conditions.size() == 1) {
            return conditions.get(0);
        }
        return Filter.and((Filter[])conditions.toArray(new Filter[0]));
    }

    protected List<Filter> getFilterList(VectorQuery query) {
        ArrayList<Filter> conditions = new ArrayList<Filter>();
        if (query.getKnowledgeIds() != null && !query.getKnowledgeIds().isEmpty()) {
            if (query.getKnowledgeIds().size() == 1) {
                conditions.add(Filter.equal((String)"knowledgeId", query.getKnowledgeIds().get(0)));
            } else {
                HashSet uniqueKnowledgeIds = new HashSet(query.getKnowledgeIds());
                Object[] knowledgeIdArray = uniqueKnowledgeIds.toArray();
                conditions.add(Filter.in((String)"knowledgeId", (Object[])knowledgeIdArray));
            }
        }
        if (query.getGroupIds() != null && !query.getGroupIds().isEmpty()) {
            if (query.getGroupIds().size() == 1) {
                conditions.add(Filter.equal((String)"groupId", query.getGroupIds().get(0)));
            } else {
                HashSet uniqueGroupIds = new HashSet(query.getGroupIds());
                Object[] groupIdArray = uniqueGroupIds.toArray();
                conditions.add(Filter.in((String)"groupId", (Object[])groupIdArray));
            }
        }
        if (query.getChunkTypes() != null && !query.getChunkTypes().isEmpty()) {
            HashSet uniqueChunkTypes = new HashSet(query.getChunkTypes());
            if (uniqueChunkTypes.size() == 1) {
                String chunkType = (String)uniqueChunkTypes.iterator().next();
                conditions.add(Filter.equal((String)"repoId", (Object)chunkType));
            } else {
                Filter[] typeFilters = new Filter[uniqueChunkTypes.size()];
                int index = 0;
                for (String chunkType : uniqueChunkTypes) {
                    typeFilters[index++] = Filter.equal((String)"repoId", (Object)chunkType);
                }
                conditions.add(Filter.or((Filter[])typeFilters));
            }
        }
        if (query.getChunkIds() != null && !query.getChunkIds().isEmpty()) {
            if (query.getChunkIds().size() == 1) {
                conditions.add(Filter.equal((String)"id", query.getChunkIds().get(0)));
            } else {
                HashSet uniqueChunkIds = new HashSet(query.getChunkIds());
                Object[] chunkIdArray = uniqueChunkIds.toArray();
                conditions.add(Filter.in((String)"id", (Object[])chunkIdArray));
            }
        }
        return conditions;
    }

    public boolean doStore(VectorChunk vectorChunk) throws VectorException {
        if (vectorChunk == null || vectorChunk.getChunk() == null) {
            logger.warn("Failed to store vector chunk: chunk is null");
            throw new IllegalArgumentException("Vector chunk cannot be null");
        }
        this.validateVector(vectorChunk);
        try {
            logger.debug("Storing single vector chunk: id={}", (Object)vectorChunk.getChunk().getId());
            this.doStore(Collections.singletonList(vectorChunk));
            logger.info("Successfully stored vector chunk: id={}", (Object)vectorChunk.getChunk().getId());
        }
        catch (Exception e) {
            logger.error("Failed to store vector chunk: id=" + vectorChunk.getChunk().getId(), (Throwable)e);
            throw new VectorException(GptasErrorCode.VECTOR_STORE_WRITE_ERROR, "Store operation failed: " + e.getMessage(), (Throwable)e);
        }
        return true;
    }

    public Pair<List<VectorChunk>, List<VectorChunk>> doBatchStore(List<VectorChunk> vectorChunks) throws VectorException {
        if (vectorChunks == null || vectorChunks.isEmpty()) {
            return new Pair(new ArrayList(1), new ArrayList(1));
        }
        logger.info("Starting batch store operation: chunk count={}", (Object)vectorChunks.size());
        for (VectorChunk vectorChunk : vectorChunks) {
            this.validateVector(vectorChunk);
        }
        ArrayList<VectorChunk> failedChunks = new ArrayList<VectorChunk>();
        ArrayList<VectorChunk> successChunks = new ArrayList<VectorChunk>();
        List<List<VectorChunk>> batches = this.partition(vectorChunks);
        logger.info("Split into {} batches, batch size={}", (Object)batches.size(), (Object)1000);
        for (List<VectorChunk> batch : batches) {
            try {
                this.doStore(batch);
                successChunks.addAll(batch);
                logger.info("Successfully stored batch: chunk count={}", (Object)batch.size());
            }
            catch (Exception e) {
                logger.error("Batch store failed: chunk count=" + batch.size(), (Throwable)e);
                failedChunks.addAll(batch);
            }
        }
        if (!failedChunks.isEmpty()) {
            logger.warn("Failed chunks: {}/{}", (Object)failedChunks.size(), (Object)vectorChunks.size());
        }
        return new Pair(successChunks, failedChunks);
    }

    public void delete(String entityId, List<Long> chunkIds) throws VectorException {
        if (chunkIds == null || chunkIds.isEmpty()) {
            logger.debug("No chunk IDs provided for deletion, skipping operation");
            return;
        }
        if (!this.initialized.get()) {
            try {
                this.initialize();
            }
            catch (VectorException e) {
                logger.error("Failed to initialize before delete operation", (Throwable)e);
                throw e;
            }
        }
        try {
            logger.debug("Deleting vector chunks: count={}, collection={}", (Object)chunkIds.size(), (Object)this.collectionName);
            this.doDelete(chunkIds);
            logger.info("Successfully deleted vector chunks: count={}, collection={}", (Object)chunkIds.size(), (Object)this.collectionName);
        }
        catch (Exception e) {
            logger.error("Failed to delete chunks: chunkIds=" + chunkIds, (Throwable)e);
            throw new VectorException(GptasErrorCode.VECTOR_STORE_DELETE_ERROR, "Delete operation failed: " + e.getMessage(), (Throwable)e);
        }
    }

    protected void doDelete(List<Long> chunkIds) {
        Object[] idArray = new Object[chunkIds.size()];
        for (int i = 0; i < chunkIds.size(); ++i) {
            idArray[i] = chunkIds.get(i);
        }
        this.vdbClient.delete(this.collectionName, idArray);
    }

    public void close() {
        try {
            logger.info("Closing vector database client");
            this.vdbClient.close();
            logger.info("Successfully closed vector database client");
        }
        catch (Exception e) {
            logger.error("Failed to close vector database client", (Throwable)e);
        }
    }

    protected void validateVector(VectorChunk vectorChunk) {
        if (vectorChunk == null) {
            throw new IllegalArgumentException("Vector chunk cannot be null");
        }
        if (vectorChunk.getVector() == null) {
            throw new IllegalArgumentException("Vector cannot be null");
        }
        if (vectorChunk.getChunk() == null) {
            throw new IllegalArgumentException("Chunk data cannot be null");
        }
        if (this.enableVector(vectorChunk.getChunk().getEntityId()) && vectorChunk.getVector().size() != this.vdbClient.getDimension()) {
            throw new IllegalArgumentException("Invalid vector dimension: expected=" + this.vdbClient.getDimension() + ", actual=" + vectorChunk.getVector().size());
        }
    }

    private void validateQuery(VectorQuery query) {
        if (query == null) {
            throw new IllegalArgumentException("Query cannot be null");
        }
        if (query.getQueryVector() == null && query.getQueryText() == null) {
            throw new IllegalArgumentException("Either query vector or query text must be provided");
        }
        if (query.getTopK() <= 0 || query.getTopK() > 1000) {
            throw new IllegalArgumentException("Invalid topK value: " + query.getTopK());
        }
    }

    private void doStore(List<VectorChunk> vectorChunks) throws VectorException {
        if (vectorChunks == null || vectorChunks.isEmpty()) {
            logger.warn("No vector chunks to store");
            return;
        }
        if (!this.initialized.get()) {
            this.initialize();
        }
        logger.info("Storing vectors: count={}, collection={}", (Object)vectorChunks.size(), (Object)this.collectionName);
        try {
            CollectionData collectionData = new CollectionData(this.collectionMeta, vectorChunks.size());
            for (VectorChunk chunk : vectorChunks) {
                if (chunk == null || chunk.getChunk() == null) {
                    logger.warn("Skipping null vector chunk");
                    continue;
                }
                collectionData.append(this.createRowData(chunk));
            }
            this.vdbClient.insert(collectionData, this.enableVector(vectorChunks.get(0).getChunk().getEntityId()));
            logger.info("Successfully stored vectors: count={}, collection={}", (Object)vectorChunks.size(), (Object)this.collectionName);
        }
        catch (Exception e) {
            logger.error("Failed to store vectors", (Throwable)e);
            throw new VectorException(GptasErrorCode.VECTOR_STORE_WRITE_ERROR, "Store operation failed: " + e.getMessage(), (Throwable)e);
        }
    }

    protected Object[] createRowData(VectorChunk chunk) {
        List floatVector = chunk.getVector();
        double[] doubleVector = new double[floatVector.size()];
        for (int i = 0; i < floatVector.size(); ++i) {
            doubleVector[i] = ((Float)floatVector.get(i)).doubleValue();
        }
        String id = chunk.getChunk().getId() != 0L ? String.valueOf(chunk.getChunk().getId()) : UUID.randomUUID().toString().replace("-", "");
        String chunkType = chunk.getChunk().getChunkType() != null ? chunk.getChunk().getChunkType() : "";
        long knowledgeId = chunk.getChunk().getKnowledgeId() != null ? chunk.getChunk().getKnowledgeId() : 0L;
        long groupId = chunk.getChunk().getGroupId() != null ? chunk.getChunk().getGroupId() : 0L;
        return new Object[]{id, doubleVector, chunkType, knowledgeId, groupId, chunk.getChunk().getContent()};
    }

    private List<VectorResult> filterResults(List<VectorResult> vectorResults, Float limitScore) {
        if (limitScore == null || vectorResults == null || vectorResults.isEmpty()) {
            return vectorResults;
        }
        ArrayList<VectorResult> filteredResults = new ArrayList<VectorResult>();
        HashSet<Long> idSet = new HashSet<Long>(16);
        for (VectorResult result : vectorResults) {
            if (idSet.contains(result.getChunk().getId())) continue;
            idSet.add(result.getChunk().getId());
            if (!(result.getScore() <= limitScore.floatValue())) continue;
            filteredResults.add(result);
        }
        return filteredResults;
    }

    private List<List<VectorChunk>> partition(List<VectorChunk> chunks) {
        if (chunks == null) {
            return Collections.emptyList();
        }
        ArrayList<List<VectorChunk>> batches = new ArrayList<List<VectorChunk>>(chunks.size());
        for (int i = 0; i < chunks.size(); i += 1000) {
            batches.add(chunks.subList(i, Math.min(chunks.size(), i + 1000)));
        }
        return batches;
    }

    public Map<String, List<?>> query(String entityId, List<Long> knowledgeIds, long start, long limit) throws VectorException {
        String[] outFields = Arrays.copyOf(OUT_FIELDS, OUT_FIELDS.length + 1);
        outFields[VDBService.OUT_FIELDS.length] = VECTOR_FIELD_NAME;
        QueryParameters queryParameters = new QueryParameters(this.getCollectionName(), (int)start, (int)limit, outFields);
        VectorQuery vectorQuery = new VectorQuery();
        vectorQuery.setKnowledgeIds(knowledgeIds);
        Filter filter = this.buildSearchFilter(vectorQuery);
        queryParameters.setFilter(filter);
        CollectionData datas = this.vdbClient.query(queryParameters);
        HashMap map = new HashMap(16);
        ArrayList<Long> ids = new ArrayList<Long>(16);
        ArrayList vectors = new ArrayList(16);
        ArrayList<Long> retKnowledgeIds = new ArrayList<Long>(16);
        map.put("id", ids);
        map.put(VECTOR_FIELD_NAME, vectors);
        map.put("knowledgeId", retKnowledgeIds);
        for (int i = 0; i < datas.size(); ++i) {
            String id = datas.getString(i, "_id");
            int pos = id.lastIndexOf("_");
            String trueId = pos > 0 ? id.substring(pos + 1) : id;
            ids.add(Long.parseLong(trueId));
            float[] vector = datas.getFloatVector(i, VECTOR_FIELD_NAME);
            ArrayList<Float> listVectors = new ArrayList<Float>(vector.length);
            for (float v : vector) {
                listVectors.add(Float.valueOf(v));
            }
            vectors.add(listVectors);
            retKnowledgeIds.add(datas.getLong(i, "knowledgeId"));
        }
        return map;
    }

    private String generateCollectionName() {
        String accountId = RequestContext.get().getAccountId();
        return String.format("KM_%s_%s_%s", accountId, this.formId, this.vdbClient.getEmbeddingName());
    }

    public void deleteAll(String entityId) {
        this.vdbClient.deleteAll(this.generateCollectionName());
    }
}

