/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.metrics;

import com.google.common.collect.Iterators;
import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;
import java.util.ServiceConfigurationError;
import java.util.ServiceLoader;
import java.util.Set;
import java.util.TreeSet;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import org.apache.flink.annotation.VisibleForTesting;
import org.apache.flink.api.java.tuple.Tuple2;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.configuration.DelegatingConfiguration;
import org.apache.flink.configuration.MetricOptions;
import org.apache.flink.core.plugin.PluginManager;
import org.apache.flink.metrics.MetricConfig;
import org.apache.flink.metrics.reporter.InstantiateViaFactory;
import org.apache.flink.metrics.reporter.InterceptInstantiationViaReflection;
import org.apache.flink.metrics.reporter.MetricReporter;
import org.apache.flink.metrics.reporter.MetricReporterFactory;
import org.apache.flink.runtime.metrics.scope.ScopeFormat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class ReporterSetup {
    private static final Logger LOG = LoggerFactory.getLogger(ReporterSetup.class);
    private static final Pattern reporterListPattern = Pattern.compile("\\s*,\\s*");
    private static final Pattern reporterClassPattern = Pattern.compile(Pattern.quote("metrics.reporter.") + "([\\S&&[^.]]*)\\." + '(' + Pattern.quote("class") + '|' + Pattern.quote("factory.class") + ')');
    private final String name;
    private final MetricConfig configuration;
    private final MetricReporter reporter;

    public ReporterSetup(String name, MetricConfig configuration, MetricReporter reporter) {
        this.name = name;
        this.configuration = configuration;
        this.reporter = reporter;
    }

    public Optional<String> getDelimiter() {
        return Optional.ofNullable(this.configuration.getString("scope.delimiter", null));
    }

    public Optional<String> getIntervalSettings() {
        return Optional.ofNullable(this.configuration.getString("interval", null));
    }

    public Set<String> getExcludedVariables() {
        String excludedVariablesList = this.configuration.getString("scope.variables.excludes", null);
        if (excludedVariablesList == null) {
            return Collections.emptySet();
        }
        HashSet<String> excludedVariables = new HashSet<String>();
        for (String exclusion : excludedVariablesList.split(";")) {
            excludedVariables.add(ScopeFormat.asVariable(exclusion));
        }
        return Collections.unmodifiableSet(excludedVariables);
    }

    public String getName() {
        return this.name;
    }

    @VisibleForTesting
    MetricConfig getConfiguration() {
        return this.configuration;
    }

    public MetricReporter getReporter() {
        return this.reporter;
    }

    @VisibleForTesting
    public static ReporterSetup forReporter(String reporterName, MetricReporter reporter) {
        return ReporterSetup.createReporterSetup(reporterName, new MetricConfig(), reporter);
    }

    @VisibleForTesting
    public static ReporterSetup forReporter(String reporterName, MetricConfig metricConfig, MetricReporter reporter) {
        return ReporterSetup.createReporterSetup(reporterName, metricConfig, reporter);
    }

    private static ReporterSetup createReporterSetup(String reporterName, MetricConfig metricConfig, MetricReporter reporter) {
        reporter.open(metricConfig);
        return new ReporterSetup(reporterName, metricConfig, reporter);
    }

    public static List<ReporterSetup> fromConfiguration(Configuration configuration, @Nullable PluginManager pluginManager) {
        String includedReportersString = configuration.getString(MetricOptions.REPORTERS_LIST, "");
        Set<String> namedReporters = ReporterSetup.findEnabledReportersInConfiguration(configuration, includedReportersString);
        if (namedReporters.isEmpty()) {
            return Collections.emptyList();
        }
        List<Tuple2<String, Configuration>> reporterConfigurations = ReporterSetup.loadReporterConfigurations(configuration, namedReporters);
        Map<String, MetricReporterFactory> reporterFactories = ReporterSetup.loadAvailableReporterFactories(pluginManager);
        return ReporterSetup.setupReporters(reporterFactories, reporterConfigurations);
    }

    private static Set<String> findEnabledReportersInConfiguration(Configuration configuration, String includedReportersString) {
        Set includedReporters = reporterListPattern.splitAsStream(includedReportersString).filter(r -> !r.isEmpty()).collect(Collectors.toSet());
        TreeSet<String> namedOrderedReporters = new TreeSet<String>(String::compareTo);
        for (String key : configuration.keySet()) {
            Matcher matcher;
            if (!key.startsWith("metrics.reporter.") || !(matcher = reporterClassPattern.matcher(key)).matches()) continue;
            String reporterName = matcher.group(1);
            if (includedReporters.isEmpty() || includedReporters.contains(reporterName)) {
                if (namedOrderedReporters.contains(reporterName)) {
                    LOG.warn("Duplicate class configuration detected for reporter {}.", (Object)reporterName);
                    continue;
                }
                namedOrderedReporters.add(reporterName);
                continue;
            }
            LOG.info("Excluding reporter {}, not configured in reporter list ({}).", (Object)reporterName, (Object)includedReportersString);
        }
        return namedOrderedReporters;
    }

    private static List<Tuple2<String, Configuration>> loadReporterConfigurations(Configuration configuration, Set<String> namedReporters) {
        ArrayList<Tuple2<String, Configuration>> reporterConfigurations = new ArrayList<Tuple2<String, Configuration>>(namedReporters.size());
        for (String namedReporter : namedReporters) {
            DelegatingConfiguration delegatingConfiguration = new DelegatingConfiguration(configuration, "metrics.reporter." + namedReporter + '.');
            reporterConfigurations.add((Tuple2<String, Configuration>)Tuple2.of((Object)namedReporter, (Object)delegatingConfiguration));
        }
        return reporterConfigurations;
    }

    private static Map<String, MetricReporterFactory> loadAvailableReporterFactories(@Nullable PluginManager pluginManager) {
        HashMap<String, MetricReporterFactory> reporterFactories = new HashMap<String, MetricReporterFactory>(2);
        Iterator<MetricReporterFactory> factoryIterator = ReporterSetup.getAllReporterFactories(pluginManager);
        while (factoryIterator.hasNext()) {
            try {
                MetricReporterFactory factory = factoryIterator.next();
                String factoryClassName = factory.getClass().getName();
                MetricReporterFactory existingFactory = (MetricReporterFactory)reporterFactories.get(factoryClassName);
                if (existingFactory == null) {
                    reporterFactories.put(factoryClassName, factory);
                    LOG.debug("Found reporter factory {} at {} ", (Object)factoryClassName, (Object)new File(factory.getClass().getProtectionDomain().getCodeSource().getLocation().toURI()).getCanonicalPath());
                    continue;
                }
                LOG.warn("Multiple implementations of the same reporter were found in 'lib' and/or 'plugins' directories for {}. It is recommended to remove redundant reporter JARs to resolve used versions' ambiguity.", (Object)factoryClassName);
            }
            catch (Exception | ServiceConfigurationError e) {
                LOG.warn("Error while loading reporter factory.", e);
            }
        }
        return Collections.unmodifiableMap(reporterFactories);
    }

    private static Iterator<MetricReporterFactory> getAllReporterFactories(@Nullable PluginManager pluginManager) {
        Iterator<MetricReporterFactory> factoryIteratorSPI = ServiceLoader.load(MetricReporterFactory.class).iterator();
        Iterator factoryIteratorPlugins = pluginManager != null ? pluginManager.load(MetricReporterFactory.class) : Collections.emptyIterator();
        return Iterators.concat((Iterator)factoryIteratorPlugins, factoryIteratorSPI);
    }

    private static List<ReporterSetup> setupReporters(Map<String, MetricReporterFactory> reporterFactories, List<Tuple2<String, Configuration>> reporterConfigurations) {
        ArrayList<ReporterSetup> reporterSetups = new ArrayList<ReporterSetup>(reporterConfigurations.size());
        for (Tuple2<String, Configuration> reporterConfiguration : reporterConfigurations) {
            String reporterName = (String)reporterConfiguration.f0;
            Configuration reporterConfig = (Configuration)reporterConfiguration.f1;
            try {
                Optional<MetricReporter> metricReporterOptional = ReporterSetup.loadReporter(reporterName, reporterConfig, reporterFactories);
                metricReporterOptional.ifPresent(reporter -> {
                    MetricConfig metricConfig = new MetricConfig();
                    reporterConfig.addAllToProperties((Properties)metricConfig);
                    reporterSetups.add(ReporterSetup.createReporterSetup(reporterName, metricConfig, reporter));
                });
            }
            catch (Throwable t) {
                LOG.error("Could not instantiate metrics reporter {}. Metrics might not be exposed/reported.", (Object)reporterName, (Object)t);
            }
        }
        return reporterSetups;
    }

    private static Optional<MetricReporter> loadReporter(String reporterName, Configuration reporterConfig, Map<String, MetricReporterFactory> reporterFactories) throws ClassNotFoundException, IllegalAccessException, InstantiationException {
        String reporterClassName = reporterConfig.getString("class", null);
        String factoryClassName = reporterConfig.getString("factory.class", null);
        if (factoryClassName != null) {
            return ReporterSetup.loadViaFactory(factoryClassName, reporterName, reporterConfig, reporterFactories);
        }
        if (reporterClassName != null) {
            Optional<MetricReporterFactory> interceptingFactory = reporterFactories.values().stream().filter(factory -> {
                InterceptInstantiationViaReflection annotation = factory.getClass().getAnnotation(InterceptInstantiationViaReflection.class);
                return annotation != null && annotation.reporterClassName().equals(reporterClassName);
            }).findAny();
            if (interceptingFactory.isPresent()) {
                return ReporterSetup.loadViaFactory(reporterConfig, interceptingFactory.get());
            }
            return ReporterSetup.loadViaReflection(reporterClassName, reporterName, reporterConfig, reporterFactories);
        }
        LOG.warn("No reporter class nor factory set for reporter {}. Metrics might not be exposed/reported.", (Object)reporterName);
        return Optional.empty();
    }

    private static Optional<MetricReporter> loadViaFactory(String factoryClassName, String reporterName, Configuration reporterConfig, Map<String, MetricReporterFactory> reporterFactories) {
        MetricReporterFactory factory = reporterFactories.get(factoryClassName);
        if (factory == null) {
            LOG.warn("The reporter factory ({}) could not be found for reporter {}. Available factories: {}.", new Object[]{factoryClassName, reporterName, reporterFactories.keySet()});
            return Optional.empty();
        }
        return ReporterSetup.loadViaFactory(reporterConfig, factory);
    }

    private static Optional<MetricReporter> loadViaFactory(Configuration reporterConfig, MetricReporterFactory factory) {
        MetricConfig metricConfig = new MetricConfig();
        reporterConfig.addAllToProperties((Properties)metricConfig);
        return Optional.of(factory.createMetricReporter((Properties)metricConfig));
    }

    private static Optional<MetricReporter> loadViaReflection(String reporterClassName, String reporterName, Configuration reporterConfig, Map<String, MetricReporterFactory> reporterFactories) throws ClassNotFoundException, IllegalAccessException, InstantiationException {
        Class<?> reporterClass = Class.forName(reporterClassName);
        InstantiateViaFactory alternativeFactoryAnnotation = reporterClass.getAnnotation(InstantiateViaFactory.class);
        if (alternativeFactoryAnnotation != null) {
            String alternativeFactoryClassName = alternativeFactoryAnnotation.factoryClassName();
            LOG.info("The reporter configuration of {} is out-dated (but still supported). Please configure a factory class instead: '{}{}.{}: {}' to ensure that the configuration continues to work with future versions.", new Object[]{reporterName, "metrics.reporter.", reporterName, "factory.class", alternativeFactoryClassName});
            return ReporterSetup.loadViaFactory(alternativeFactoryClassName, reporterName, reporterConfig, reporterFactories);
        }
        return Optional.of((MetricReporter)reporterClass.newInstance());
    }
}

