/*
 * Decompiled with CFR 0.152.
 */
package org.decimal4j.arithmetic;

import org.decimal4j.api.DecimalArithmetic;
import org.decimal4j.arithmetic.Div;
import org.decimal4j.arithmetic.DoubleConversion;
import org.decimal4j.arithmetic.Rounding;
import org.decimal4j.arithmetic.Unsigned;
import org.decimal4j.scale.ScaleMetrics;
import org.decimal4j.truncate.DecimalRounding;
import org.decimal4j.truncate.TruncatedPart;

final class FloatConversion {
    private static final long LONG_MASK = 0xFFFFFFFFL;
    private static final int SIGNIFICAND_MASK = 0x7FFFFF;
    private static final int EXPONENT_MASK = 2139095040;
    private static final int SIGN_MASK = Integer.MIN_VALUE;
    private static final int SIGNIFICAND_BITS = 23;
    private static final int EXPONENT_BIAS = 127;
    private static final int IMPLICIT_BIT = 0x800000;
    private static final float MIN_LONG_AS_FLOAT = -9.223372E18f;
    private static final float MAX_LONG_AS_FLOAT_PLUS_ONE = 9.223372E18f;

    public static final long floatToLong(float value) {
        if (Float.isNaN(value)) {
            throw new IllegalArgumentException("Cannot convert float to long: " + value);
        }
        if (FloatConversion.isInLongRange(value)) {
            return (long)value;
        }
        throw new IllegalArgumentException("Overflow for conversion from float to long: " + value);
    }

    public static final long floatToLong(DecimalRounding rounding, float value) {
        if (Float.isNaN(value)) {
            throw new IllegalArgumentException("Cannot convert float to long: " + value);
        }
        if (FloatConversion.isInLongRange(value)) {
            return (long)FloatConversion.roundIntermediate(value, rounding);
        }
        throw new IllegalArgumentException("Overflow for conversion from float to long: " + value);
    }

    private static final float roundIntermediate(float x, DecimalRounding mode) {
        switch (mode) {
            case UNNECESSARY: {
                if (!FloatConversion.isMathematicalInteger(x)) {
                    throw new ArithmeticException("Rounding necessary to convert to an integer value: " + x);
                }
                return x;
            }
            case FLOOR: {
                if (x >= 0.0f || FloatConversion.isMathematicalInteger(x)) {
                    return x;
                }
                return (long)x - 1L;
            }
            case CEILING: {
                if (x <= 0.0f || FloatConversion.isMathematicalInteger(x)) {
                    return x;
                }
                return (long)x + 1L;
            }
            case DOWN: {
                return x;
            }
            case UP: {
                if (FloatConversion.isMathematicalInteger(x)) {
                    return x;
                }
                return (long)x + (x > 0.0f ? 1L : -1L);
            }
            case HALF_EVEN: {
                return FloatConversion.rint(x);
            }
            case HALF_UP: {
                float z = FloatConversion.rint(x);
                if (Math.abs(x - z) == 0.5f) {
                    return x + Math.copySign(0.5f, x);
                }
                return z;
            }
            case HALF_DOWN: {
                float z = FloatConversion.rint(x);
                if (Math.abs(x - z) == 0.5f) {
                    return x;
                }
                return z;
            }
        }
        throw new IllegalArgumentException("Unsupported rounding mode: " + (Object)((Object)mode));
    }

    public static final long floatToUnscaled(DecimalArithmetic arith, float value) {
        return FloatConversion.floatToUnscaled(arith, DecimalRounding.DOWN, value);
    }

    public static final long floatToUnscaled(DecimalArithmetic arith, DecimalRounding rounding, float value) {
        if (value == 0.0f) {
            return 0L;
        }
        int exp = Math.getExponent(value);
        if (exp >= 64) {
            throw FloatConversion.newOverflowException(arith, value);
        }
        ScaleMetrics scaleMetrics = arith.getScaleMetrics();
        long significand = FloatConversion.getSignificand(value);
        int lFactor = (int)(significand & 0xFFFFFFFFL);
        int hFactor = (int)(significand >>> 32);
        long t = scaleMetrics.mulloByScaleFactor(lFactor);
        long w3 = t & 0xFFFFFFFFL;
        long k = t >>> 32;
        t = scaleMetrics.mulloByScaleFactor(hFactor) + k;
        long w2 = t & 0xFFFFFFFFL;
        long w1 = t >>> 32;
        t = scaleMetrics.mulhiByScaleFactor(lFactor) + w2;
        k = t >>> 32;
        long hScaled = scaleMetrics.mulhiByScaleFactor(hFactor) + w1 + k;
        long lScaled = (t << 32) + w3;
        int shift = exp - 23;
        return FloatConversion.floatToUnscaledShift(arith, rounding, value, hScaled, lScaled, shift);
    }

    private static final long floatToUnscaledShift(DecimalArithmetic arith, DecimalRounding rounding, float value, long hScaled, long lScaled, int shift) {
        if (shift > 0) {
            if (hScaled != 0L) {
                throw FloatConversion.newOverflowException(arith, value);
            }
            int zeros = Long.numberOfLeadingZeros(lScaled);
            if (shift >= zeros) {
                throw FloatConversion.newOverflowException(arith, value);
            }
            long absResult = lScaled << shift;
            return value >= 0.0f ? absResult : -absResult;
        }
        if (shift == 0) {
            if (hScaled != 0L | lScaled < 0L) {
                throw FloatConversion.newOverflowException(arith, value);
            }
            return value >= 0.0f ? lScaled : -lScaled;
        }
        if (rounding == DecimalRounding.DOWN) {
            return FloatConversion.floatToUnscaledShiftRight(arith, value, hScaled, lScaled, -shift);
        }
        return FloatConversion.floatToUnscaledShiftRight(arith, rounding, value, hScaled, lScaled, -shift);
    }

    private static final long floatToUnscaledShiftRight(DecimalArithmetic arith, float value, long hScaled, long lScaled, int shift) {
        long absResult;
        if (shift < 64) {
            if (hScaled >>> shift != 0L) {
                throw FloatConversion.newOverflowException(arith, value);
            }
            absResult = hScaled << 64 - shift | lScaled >>> shift;
        } else if (shift < 128) {
            absResult = hScaled >>> shift - 64;
        } else {
            return 0L;
        }
        if (absResult < 0L) {
            throw FloatConversion.newOverflowException(arith, value);
        }
        return value >= 0.0f ? absResult : -absResult;
    }

    private static final long floatToUnscaledShiftRight(DecimalArithmetic arith, DecimalRounding rounding, float value, long hScaled, long lScaled, int shift) {
        int inc;
        TruncatedPart truncatedPart;
        long rem;
        long absResult;
        if (shift < 64) {
            if (hScaled >>> shift != 0L) {
                throw FloatConversion.newOverflowException(arith, value);
            }
            absResult = hScaled << 64 - shift | lScaled >>> shift;
            rem = FloatConversion.modPow2(lScaled, shift);
            truncatedPart = Rounding.truncatedPartFor2powN(rem, shift);
        } else if (shift < 128) {
            absResult = hScaled >>> shift - 64;
            rem = FloatConversion.modPow2(hScaled, shift - 64);
            truncatedPart = Rounding.truncatedPartFor2powN(rem, lScaled, shift);
        } else {
            absResult = 0L;
            truncatedPart = Rounding.truncatedPartFor2powN(hScaled, lScaled, shift);
        }
        int n = inc = absResult < 0L ? 0 : rounding.calculateRoundingIncrement(value >= 0.0f ? 1 : -1, absResult, truncatedPart);
        if (absResult < 0L | value >= 0.0f & absResult == Long.MAX_VALUE & inc == 1) {
            throw FloatConversion.newOverflowException(arith, value);
        }
        return (value >= 0.0f ? absResult : -absResult) + (long)inc;
    }

    public static final float longToFloat(DecimalArithmetic arith, long value) {
        return FloatConversion.unscaledToFloat(arith, DecimalRounding.DOWN, value);
    }

    public static final float longToFloat(DecimalArithmetic arith, DecimalRounding rounding, long value) {
        if (rounding == DecimalRounding.HALF_EVEN) {
            return value;
        }
        return FloatConversion.unscaledToFloat(arith, rounding, value);
    }

    public static final float unscaledToFloat(DecimalArithmetic arith, long unscaled) {
        return FloatConversion.unscaledToFloat(arith, DecimalRounding.DOWN, unscaled);
    }

    public static final float unscaledToFloat(DecimalArithmetic arith, DecimalRounding rounding, long unscaled) {
        int mantissaShift;
        long valModFactor;
        int exp;
        int pow2;
        if (unscaled == 0L) {
            return 0.0f;
        }
        if (rounding == DecimalRounding.HALF_EVEN) {
            return (float)DoubleConversion.unscaledToDouble(arith, rounding, unscaled);
        }
        ScaleMetrics scaleMetrics = arith.getScaleMetrics();
        long absUnscaled = Math.abs(unscaled);
        long absVal = absUnscaled >>> (pow2 = Long.numberOfTrailingZeros(absUnscaled));
        int nlzAbsVal = Long.numberOfLeadingZeros(absVal);
        int alignShift = nlzAbsVal - scaleMetrics.getScaleFactorNumberOfLeadingZeros();
        if (alignShift >= 0) {
            long scaledAbsVal = absVal << alignShift;
            long diff = scaledAbsVal - scaleMetrics.getScaleFactor();
            exp = -alignShift + (int)(diff >> 63);
            valModFactor = diff + (diff >> 63 & scaledAbsVal);
            mantissaShift = 23;
        } else {
            long scaledFactor = scaleMetrics.getScaleFactor() << -alignShift;
            if (Unsigned.isLess(absVal, scaledFactor)) {
                exp = -alignShift - 1;
                valModFactor = absVal - (scaledFactor >>> 1);
                mantissaShift = 23 + alignShift + 1;
            } else {
                exp = -alignShift;
                valModFactor = absVal - scaledFactor;
                mantissaShift = 23 + alignShift;
            }
        }
        if (rounding == DecimalRounding.DOWN) {
            return FloatConversion.unscaledToFloatShiftAndDivideByScaleFactor(scaleMetrics, unscaled, exp + pow2, mantissaShift, valModFactor);
        }
        return FloatConversion.unscaledToFloatShiftAndDivideByScaleFactor(scaleMetrics, rounding, unscaled, exp + pow2, mantissaShift, valModFactor);
    }

    private static final float unscaledToFloatShiftAndDivideByScaleFactor(ScaleMetrics scaleMetrics, long unscaled, int exp, int mantissaShift, long valModFactor) {
        long quot;
        if (mantissaShift >= 0) {
            long hValModFactor = valModFactor >>> 64 - mantissaShift & (long)(-mantissaShift >> 63);
            long lValModFactor = valModFactor << mantissaShift;
            quot = hValModFactor == 0L ? scaleMetrics.divideUnsignedByScaleFactor(lValModFactor) : Math.abs(Div.div128by64(DecimalRounding.DOWN, unscaled < 0L, hValModFactor, lValModFactor, scaleMetrics.getScaleFactor()));
        } else {
            quot = scaleMetrics.divideByScaleFactor(valModFactor >>> -mantissaShift);
        }
        int signBit = (int)(unscaled >>> 32 & Integer.MIN_VALUE);
        int raw = signBit | exp + 127 << 23 | (int)(quot & 0x7FFFFFL);
        return Float.intBitsToFloat(raw);
    }

    private static final float unscaledToFloatShiftAndDivideByScaleFactor(ScaleMetrics scaleMetrics, DecimalRounding rounding, long unscaled, int exp, int mantissaShift, long valModFactor) {
        long quotient;
        long scaleFactor = scaleMetrics.getScaleFactor();
        if (mantissaShift >= 0) {
            long hValModFactor = valModFactor >>> 64 - mantissaShift & (long)(-mantissaShift >> 63);
            long lValModFactor = valModFactor << mantissaShift;
            if (hValModFactor == 0L) {
                long truncated = scaleMetrics.divideUnsignedByScaleFactor(lValModFactor);
                long remainder = FloatConversion.applySign(unscaled, lValModFactor - scaleMetrics.multiplyByScaleFactor(truncated));
                quotient = truncated + (long)Math.abs(Rounding.calculateRoundingIncrementForDivision(rounding, truncated, remainder, scaleFactor));
            } else {
                quotient = Math.abs(Div.div128by64(rounding, unscaled < 0L, hValModFactor, lValModFactor, scaleFactor));
            }
        } else {
            long scaledVal = valModFactor >>> -mantissaShift;
            long truncated = scaleMetrics.divideByScaleFactor(scaledVal);
            long remainder = FloatConversion.applySign(unscaled, scaledVal - scaleMetrics.multiplyByScaleFactor(truncated) << -mantissaShift | valModFactor & -1L >>> 64 + mantissaShift);
            long shiftedScaleFactor = -mantissaShift >= scaleMetrics.getScaleFactorNumberOfLeadingZeros() ? Long.MAX_VALUE : scaleFactor << -mantissaShift;
            quotient = truncated + (long)Math.abs(Rounding.calculateRoundingIncrementForDivision(rounding, truncated, remainder, shiftedScaleFactor));
        }
        int signBit = (int)(unscaled >>> 32 & Integer.MIN_VALUE);
        int raw = quotient <= 0x7FFFFFL ? signBit | exp + 127 << 23 | (int)(quotient & 0x7FFFFFL) : signBit | exp + 1 + 127 << 23;
        return Float.intBitsToFloat(raw);
    }

    private static final long modPow2(long value, int n) {
        return value & -1L >>> 64 - n & (long)(-n >> 31);
    }

    private static final long applySign(long signed, long value) {
        return signed >= 0L ? value : -value;
    }

    private static final boolean isInLongRange(float value) {
        return -9.223372E18f - value < 1.0f & value < 9.223372E18f;
    }

    private static final boolean isMathematicalInteger(float x) {
        return FloatConversion.isFinite(x) && (x == 0.0f || 23 - Long.numberOfTrailingZeros(FloatConversion.getSignificand(x)) <= Math.getExponent(x));
    }

    private static final boolean isFinite(float f) {
        return Math.abs(f) <= Float.MAX_VALUE;
    }

    private static final int getSignificand(float f) {
        int exponent = Math.getExponent(f);
        int bits = Float.floatToRawIntBits(f);
        return exponent == -127 ? bits << 1 : (bits &= 0x7FFFFF) | 0x800000;
    }

    private static final float rint(float a) {
        float twoToThe23 = 8388608.0f;
        float sign = Math.copySign(1.0f, a);
        if ((a = Math.abs(a)) < twoToThe23) {
            a = twoToThe23 + a - twoToThe23;
        }
        return sign * a;
    }

    private static final IllegalArgumentException newOverflowException(DecimalArithmetic arith, double value) {
        return new IllegalArgumentException("Overflow for conversion from float to decimal with scale " + arith.getScale() + ": " + value);
    }

    private FloatConversion() {
    }
}

