function trans_inherit({ types: t }) {
    return {
        visitor: {
            ImportDeclaration(path) {
                if (path.node.specifiers.length > 0) {
                    path.node.specifiers.forEach(specifier => {
                        const importStatement = t.stringLiteral(path.getSource()).value;
                        if (t.isImportDefaultSpecifier(specifier)) {
                            //return `${specifier.local.name} (default)`;
                            throw new Error('Unsupported import statement: ' + path)
                        } else if (t.isImportSpecifier(specifier)) {
                            //return `${specifier.imported.name} as ${specifier.local.name}`;
                            Context.addImportSpecifier(importStatement, specifier.imported.name, specifier.local.name);
                        } else if (t.isImportNamespaceSpecifier(specifier)) {
                            //return `${specifier.local.name} (*)`;
                            Context.addImportSpecifier(importStatement, null, specifier.local.name);
                        }
                    });
                }
            },

            ClassDeclaration(path) {
                const { node } = path;

                //提取父类(含接口)
                let supers = [];
                if (node.superClass) {
                    supers.push(node.superClass.name);
                }
                if (node.implements) {
                    for (var i = 0; i < node.implements.length; i++) {
                        supers.push(node.implements[i].expression.name);
                    }
                }
                const superClassesAndInterfaces = supers.map(e => t.identifier(e));

                // extend java class?
                let extendJava = superClassesAndInterfaces.length > 0 && Context.isJavaType(superClassesAndInterfaces[0].name);

                if (extendJava) {

                    // 移除declare语句
                    path.node.body.body = path.node.body.body.filter(member => !member.declare);

                    // 提取类名
                    const className = node.id.name;

                    //1.翻译super
                    const body = path.node.body.body;
                    body.forEach(member => {
                        if (t.isFunction(member)) {
                            // 遍历方法体中的每一个节点
                            path.traverse({
                                // 访问Super节点
                                Super(superPath) {
                                    // 如果是Super()（即super的构造函数调用）
                                    if (t.isCallExpression(superPath.parent)) {
                                        // super替换为Super(...)
                                        superPath.replaceWith(t.identifier('Super'));
                                    }
                                    // 如果是MemberExpression（即super.method()）
                                    else if (t.isMemberExpression(superPath.parent)) {
                                        // 替换为this.super.method()
                                        superPath.replaceWith(t.identifier('this.super'));
                                    }
                                }
                            });
                        }
                    });


                    //2.翻译继承

                    // 提取类的方法和其他属性
                    const classBody = node.body.body.map(p => {
                        if (p.static) {
                            // 不支持静态方法，无法对等翻译：
                            // 可以实现静态属性，但无法实现多个实例之间共享静态属性(实际可以实现，但模拟成本太大，overrides部分需要全局缓存且不可丢弃，对于ts这种模块文件，临时类多，缓存会爆掉)
                            // @see ScriptExtendJava#read/writeMember、HostAdapterServices#getClassOverrides
                            throw new Error('Unsupported static member: ' + p.key.name)
                        }

                        // 属性或方法
                        return t.objectProperty(
                            t.identifier(p.key.name),
                            t.isClassProperty(p) ? p.value : t.functionExpression(null, p.params, p.body)
                        );
                    });

                    //重写getPluginName方法
                    let scriptPath = Context.getScriptPath();
                    if (scriptPath) {
                        let hasGetPluginName = false;
                        // 遍历类体中的所有成员
                        node.body.body.forEach(member => {
                            // 检查当前成员是否为方法
                            if (t.isClassMethod(member) && member.kind === 'method') {
                                // 检查方法名是否为 'getPluginName'
                                if (member.key.name === 'getPluginName') {
                                    hasGetPluginName = true;
                                }
                            }
                        });
                        if (!hasGetPluginName) {
                            let np = t.objectProperty(
                                t.identifier('getPluginName'),
                                t.functionExpression(
                                    null,
                                    [],
                                    t.blockStatement([t.returnStatement(t.stringLiteral(scriptPath))])
                                )
                            );
                            classBody.push(np);
                        }
                    }

                    // 构建 $.extend 调用
                    const extendCall = t.callExpression(
                        t.memberExpression(t.identifier('$'), t.identifier('extend')),
                        [...superClassesAndInterfaces, t.objectExpression(classBody)]
                    );

                    // 替换类声明为变量声明
                    path.replaceWith(t.variableDeclaration('const', [
                        t.variableDeclarator(t.identifier(className), extendCall)
                    ]));
                } else {
                    //翻译继承
                    // 提取类的方法和其他属性
                    const classBody = node.body.body.map(p => {
                        if (p.static) {
                            // 不支持静态方法，无法对等翻译：
                            // 可以实现静态属性，但无法实现多个实例之间共享静态属性(实际可以实现，但模拟成本太大，overrides部分需要全局缓存且不可丢弃，对于ts这种模块文件，临时类多，缓存会爆掉)
                            // @see ScriptExtendJava#read/writeMember、HostAdapterServices#getClassOverrides
                            throw new Error('Unsupported static member: ' + p.key.name)
                        }
                    });
                }
            }
        }
    }
};

Babel.registerPlugin("trans_inherit", trans_inherit);
