package kd.bos.ext.imc.formplugin.botp;
import kd.bos.dataentity.resource.ResManager;

import com.google.common.collect.Lists;
import kd.bos.dataentity.entity.DynamicObject;
import kd.bos.dataentity.entity.DynamicObjectCollection;
import kd.bos.dataentity.serialization.SerializationUtils;
import kd.bos.dataentity.utils.StringUtils;
import kd.bos.entity.ExtendedDataEntity;
import kd.bos.entity.ExtendedDataEntitySet;
import kd.bos.entity.botp.plugin.AbstractConvertPlugIn;
import kd.bos.entity.botp.plugin.args.AfterConvertEventArgs;
import kd.bos.exception.KDBizException;
import kd.bos.ext.imc.common.constant.*;
import kd.bos.ext.imc.common.util.*;
import kd.bos.logging.Log;
import kd.bos.logging.LogFactory;
import kd.bos.orm.query.QCP;
import kd.bos.orm.query.QFilter;
import kd.bos.orm.util.CollectionUtils;
import kd.bos.servicehelper.BusinessDataServiceHelper;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 上游botp推送开票申请单时，botp计算税额插件
 *
 上游botp推送开票申请单时，增加botp计算税额插件，当上游传输物料，商品名称，税收分类编码，含税单价和价税合计
 1、税额、不含税金额计算：
 （1）没传税率、传了税编，按照税编获取税率，并且计算税额、不含税金额，税率转成开票申请单下拉的格式。
 （2）没传税率、税编，报错提示：缺少税收分类编码、税率，请补充
 明细税额、不含税金额计算后，需更新剩余可开金额，需累加到表头。
 2、数量、单价计算
 （1）已传含税单价，金额（按步骤1计算或上游传值），数量=含税金额/含税单价，不含税单价=含税单价/（1+税率）
 （2）已传不含税单价，金额（按步骤1计算或上游传值），数量=不含税金额/不含税单价，含税单价=不含税单价*（1+税率）
 */
public class UpstreamToOriginalPlugin extends AbstractConvertPlugIn {
    private static final Log LOGGER = LogFactory.getLog(UpstreamToOriginalPlugin.class);

    @Override
    public void afterConvert(AfterConvertEventArgs e) {
        long startTime = System.currentTimeMillis();
        ExtendedDataEntitySet entitySet = e.getTargetExtDataEntitySet();
        ExtendedDataEntity[] entities = entitySet.FindByEntityKey(OriginalBillConstant.FORM_ID);
        if(entities==null || entities.length==0){
            throw new KDBizException(ResManager.loadKDString("botp下推的开票申请单数据为空", "UpstreamToOriginalPlugin_0", "bos-ext-imc"));
        }

        List<DynamicObject> originalBillList = Lists.newArrayListWithCapacity(entities.length);
        //通过传输的税收分类编码获取对应的税编信息
        for (ExtendedDataEntity entity : entities) {
            //接收到的单据数据
            DynamicObject originalBill = entity.getDataEntity();
            String billNo = originalBill.getString(OriginalBillConstant.BILLNO);
            LOGGER.info(String.format("billNo:%s,UpstreamToOriginalBotpConvertJson:%s", billNo, SerializationUtils.toJsonString(originalBill)));
            DynamicObjectCollection originalBillItems = originalBill.getDynamicObjectCollection(OriginalBillConstant.ITEM_KEY);
            if(CollectionUtils.isEmpty(originalBillItems)){
                throw new KDBizException(String.format(ResManager.loadKDString("单据%s没有明细数据，不能下推开票申请单", "UpstreamToOriginalPlugin_1", "bos-ext-imc"), billNo));
            }
            originalBillList.add(originalBill);
        }

        Map<Long, List<DynamicObject>> orgBills = originalBillList.stream().collect(Collectors.groupingBy(k -> DynamicObjectUtil.getDynamicObjectLongValue(k.get(OriginalBillConstant.ORG_ID))));
        String batchNo = "UpstreamConvert_" + UUID.randomUUID();
        orgBills.forEach((orgId, bills) -> originalBillBotpConvertBatch(orgId, bills, batchNo));
        originalBillList.stream().collect(Collectors.toMap(k -> k.getString(OriginalBillConstant.BILLNO), v -> v, (v1, v2) -> v2));
        long endTime = System.currentTimeMillis();
        LOGGER.info("UpstreamToOriginalPlugin处理时间为:%s", (endTime - startTime));
    }

    private static void originalBillBotpConvertBatch(long orgId, List<DynamicObject> originalBillList, String batchNo){
        LOGGER.info(String.format("执行的批次为：%s", batchNo));
        long startTime = System.currentTimeMillis();
        if (CollectionUtils.isEmpty(originalBillList)) {
            throw new KDBizException(ResManager.loadKDString("botp下推参数为空", "UpstreamToOriginalPlugin_2", "bos-ext-imc"));
        }
        if (orgId == 0L) {
            throw new KDBizException(ResManager.loadKDString("下推数据没有组织，请先配置组织", "UpstreamToOriginalPlugin_3", "bos-ext-imc"));
        }
        DynamicObject org = BusinessDataServiceHelper.loadSingle(CommonConstant.BDM_ORG, CommonConstant.ORG_FIELD_EPINFO, new QFilter(CommonConstant.ID, QFilter.equals, orgId).toArray());
        if (org == null) {
            throw new KDBizException(ResManager.loadKDString("下推的组织在发票云没有引入，请先引入", "UpstreamToOriginalPlugin_4", "bos-ext-imc"));
        }
        DynamicObject epInfo = org.getDynamicObject(CommonConstant.ORG_FIELD_EPINFO);
        if (epInfo == null) {
            throw new KDBizException(ResManager.loadKDString("下推的组织还未完善企业名称、税号等信息，请先到“发票云-基础资料-企业管理”完善资料", "UpstreamToOriginalPlugin_5", "bos-ext-imc"));
        }

        //通过明细行获取税收分类编码数据
        Map<String, DynamicObject> initTaxCodeMap = initTaxCodeMap(originalBillList);
        for(DynamicObject originalBill : originalBillList) {
            //明细行处理
            handleOriginalItem(initTaxCodeMap, originalBill);
        }
        long endTime = System.currentTimeMillis();
        LOGGER.info(String.format("批次：%s, 消耗时间为：%s", batchNo, (endTime-startTime)));
    }


    private static void handleOriginalItem(Map<String, DynamicObject> initTaxCodeMap, DynamicObject originalBill){
        String billNo = originalBill.getString(OriginalBillConstant.BILLNO);
        String taxFlag = originalBill.getString(OriginalBillConstant.HSBZ);
        DynamicObjectCollection itemList = originalBill.getDynamicObjectCollection(OriginalBillConstant.ITEM_KEY);
        if(CollectionUtils.isEmpty(itemList)){
            return;
        }

        boolean isTaxRate = false;
        for(int i=0;i<itemList.size();i++){
            DynamicObject billItem = itemList.get(i);
            //行性质
            String rowType = billItem.getString(OriginalBillConstant.Item.ROWTYPE);
            //税率
            String taxRate = billItem.getString(OriginalBillConstant.Item.TAXRATE);
            //税收分类编码
            String goodsCode = billItem.getString(OriginalBillConstant.Item.GOODSCODE);
            if (StringUtils.equals(rowType, OriginalBillConstant.RowTypeEnum.DISCOUNT)) {
                continue;
            }
            if (StringUtils.isEmpty(taxRate)) {
                if (StringUtils.isEmpty(goodsCode)) {
                    //没传税率、税编，走开票项匹配，获取税编、税率，计算税额、不含税金额。
                    LOGGER.info("暂时搁置不实现...");
                    throw new KDBizException(String.format(ResManager.loadKDString("单据编号：%s，缺少税收分类编码、税率，请补充", "UpstreamToOriginalPlugin_6", "bos-ext-imc"), billNo));
                } else {
                    // 没传税率、传了税编，按照税编获取税率，并且计算税额、不含税金额，税率转成开票申请单下拉的格式。
                    String taxRateStr = setDetailItemByGoodsCode(initTaxCodeMap, goodsCode);
                    if(StringUtils.isEmpty(taxRateStr)){
                        throw new KDBizException(String.format(ResManager.loadKDString("单据编号：%1$s，第【%2$s】的税收分类编码【%3$s】未匹配到税率",
                                "UpstreamToOriginalPlugin_7", "bos-ext-imc"), billNo, i+1, goodsCode));
                    }
                    isTaxRate = true;
                    billItem.set(OriginalBillConstant.Item.TAXRATE, taxRateStr);
                }
            }
        }

        if(isTaxRate){
            BigDecimal invoiceAmount = BigDecimal.ZERO;
            BigDecimal taxAmount = BigDecimal.ZERO;
            for(int i=0;i<itemList.size();i++){
                DynamicObject billItem = itemList.get(i);
                setItemAmountInfo(billItem);
                checkAmountPriceQuantity(billItem, i+1, billNo, taxFlag);

                if(!MathUtils.isNullOrZero(billItem.getBigDecimal(OriginalBillConstant.Item.AMOUNT))){
                    invoiceAmount = invoiceAmount.add(billItem.getBigDecimal(OriginalBillConstant.Item.AMOUNT));
                }
                if(!MathUtils.isNullOrZero(billItem.getBigDecimal(OriginalBillConstant.Item.TAX))){
                    taxAmount = taxAmount.add(billItem.getBigDecimal(OriginalBillConstant.Item.TAX));
                }
            }

            originalBill.set(OriginalBillConstant.SURPLUS_AMOUNT, invoiceAmount);
            originalBill.set(OriginalBillConstant.SURPLUS_TAX, taxAmount);
            BigDecimal totalAmount = invoiceAmount.add(taxAmount);
            originalBill.set(OriginalBillConstant.OLDTOTALAMOUNT, totalAmount);
            originalBill.set(OriginalBillConstant.TOTALAMOUNT, totalAmount);
            originalBill.set(OriginalBillConstant.INVOICEAMOUNT, invoiceAmount);
            originalBill.set(OriginalBillConstant.TOTALTAX, taxAmount);
        }
    }

    private static void checkAmountPriceQuantity(DynamicObject detail, int index, String billNo, String taxFlag) {
        String rowType = detail.getString(OriginalBillConstant.Item.ROWTYPE);
        if (StringUtils.equals(rowType, OriginalBillConstant.RowTypeEnum.DISCOUNT)) {
            return;
        }
        if(StringUtils.equals(taxFlag, OriginalBillConstant.HSBZEnum.NO_INCLUDE_TAX)){
            //不含税
            setAmountPriceQuantityBHS(detail, index, billNo);
        } else{
            //含税
            setAmountPriceQuantityHS(detail, index, billNo);
        }
    }

    /**
     * 含税明细行处理
     * @param detail
     * @param index
     * @param billNo
     */
    private static void setAmountPriceQuantityBHS(DynamicObject detail, int index, String billNo){
        LOGGER.info(String.format("通过不含税设置对应的信息：%s, 明细为：%s, index为：", billNo, SerializationUtils.toJsonString(detail), index));
        BigDecimal unitPrice = detail.getBigDecimal(OriginalBillConstant.Item.UNITPRICE);  //不含税
        BigDecimal num = detail.getBigDecimal(OriginalBillConstant.Item.NUM);
        BigDecimal amount = detail.getBigDecimal(OriginalBillConstant.Item.AMOUNT);

        //当金额为空 单价或数量有一个为空时 数据异常
        if (MathUtils.isNullOrZero(amount) && !MathUtils.isNullOrZero(num) && !MathUtils.isNullOrZero(unitPrice)) {
            //当金额为空数量和单价不为空时反算金额
            BigDecimal calAmount = unitPrice.multiply(num).setScale(2, BigDecimal.ROUND_HALF_UP);
            detail.set(OriginalBillConstant.Item.AMOUNT, calAmount);
        } else if (!MathUtils.isNullOrZero(amount) && !MathUtils.isNullOrZero(num) && !MathUtils.isNullOrZero(unitPrice)) {
            LOGGER.info("金额，数量，单价都传输的情况...");
        } else if (!MathUtils.isNullOrZero(amount) && !MathUtils.isNullOrZero(num) && MathUtils.isNullOrZero(unitPrice)) {
            //传数量金额，未传单价
            try {
                BigDecimal price = UnitPriceHelper.calcPriceOrNum(amount, num);
                detail.set(OriginalBillConstant.Item.UNITPRICE, price);
            } catch (KDBizException e) {
                throw new KDBizException(String.format(ResManager.loadKDString("单据编号[%1$s]第[%2$s]行[%3$s]", "UpstreamToOriginalPlugin_8", "bos-ext" +
                        "-imc"), billNo, index, e.getMessage()));
            }
        } else if (!MathUtils.isNullOrZero(amount) && MathUtils.isNullOrZero(num) && !MathUtils.isNullOrZero(unitPrice)) {
            //传单价金额，未传数量
            try {
                BigDecimal quantity = UnitPriceHelper.calcPriceOrNum(amount, unitPrice);
                detail.set(OriginalBillConstant.Item.NUM, quantity);
            } catch (KDBizException e) {
                throw new KDBizException(String.format(ResManager.loadKDString("单据编号[%1$s]第[%2$s]行[%3$s]", "UpstreamToOriginalPlugin_8", "bos-ext-imc"), billNo, index, e.getMessage()));
            }
        } else{
            LOGGER.info(String.format("单据编号【%s】传输的不含税明细行数据存在问题", SerializationUtils.toJsonString(detail)));
            throw new KDBizException(String.format(String.format(ResManager.loadKDString("单据编号[%1$s]第[%2$s]行，不含税情况下，不含税单价、不含税金额、数量不能都不传输",
                    "UpstreamToOriginalPlugin_9", "bos-ext-imc"), billNo, index)));
        }

        BigDecimal taxRate = new BigDecimal(detail.getString(OriginalBillConstant.Item.TAXRATE));
        BigDecimal taxUnitPrice = detail.getBigDecimal(OriginalBillConstant.Item.TAXUNITPRICE);
        BigDecimal tax = detail.getBigDecimal(OriginalBillConstant.Item.TAX);
        //税率有值
        if(MathUtils.isNullOrZero(tax)){
            BigDecimal calTax = amount.multiply(taxRate).setScale(2, BigDecimal.ROUND_HALF_UP);
            detail.set(OriginalBillConstant.Item.TAX, calTax);
        }
        //计算含税单价
        if(MathUtils.isNullOrZero(taxUnitPrice)
                || (unitPrice!=null && unitPrice.compareTo(taxUnitPrice)==0)){
            BigDecimal includingPrice = UnitPriceHelper.getTaxUnitPriceByBHS(taxRate, detail.getBigDecimal(OriginalBillConstant.Item.NUM),
                    null, detail.getBigDecimal(OriginalBillConstant.Item.UNITPRICE));
            detail.set(OriginalBillConstant.Item.TAXUNITPRICE, includingPrice);
        }
        BigDecimal taxAmount = detail.getBigDecimal(OriginalBillConstant.Item.TAXAMOUNT);
        if(MathUtils.isNullOrZero(taxAmount)
                || (amount!=null && taxAmount.compareTo(amount)==0)){
            BigDecimal includingAmount = detail.getBigDecimal(OriginalBillConstant.Item.AMOUNT).add(detail.getBigDecimal(OriginalBillConstant.Item.TAX)).setScale(2, BigDecimal.ROUND_HALF_UP);
            detail.set(OriginalBillConstant.Item.TAXAMOUNT, includingAmount);
        }

        detail.set(OriginalBillConstant.Item.REMAINVALIDTAX, detail.getBigDecimal(OriginalBillConstant.Item.TAX));
        detail.set(OriginalBillConstant.Item.REMAINVALIDAMOUNT, detail.getBigDecimal(OriginalBillConstant.Item.AMOUNT));
        detail.set(OriginalBillConstant.Item.REMAINVALIDNUM, detail.getBigDecimal(OriginalBillConstant.Item.NUM));
    }

    /**
     * 不含税明细行处理
     * @param detail
     * @param index
     * @param billNo
     */
    private static void setAmountPriceQuantityHS(DynamicObject detail, int index, String billNo){
        LOGGER.info(String.format("通过含税设置对应的信息：%s, 明细为：%s, index为：%s", billNo, SerializationUtils.toJsonString(detail), index));
        BigDecimal taxUnitPrice = detail.getBigDecimal(OriginalBillConstant.Item.TAXUNITPRICE);  //含税单价
        BigDecimal num = detail.getBigDecimal(OriginalBillConstant.Item.NUM);
        BigDecimal taxAmount = detail.getBigDecimal(OriginalBillConstant.Item.TAXAMOUNT);

        //当金额为空 单价或数量有一个为空时 数据异常
        if (MathUtils.isNullOrZero(taxAmount) && !MathUtils.isNullOrZero(num) && !MathUtils.isNullOrZero(taxUnitPrice)) {
            //当金额为空数量和单价不为空时反算金额
            BigDecimal calAmount = taxUnitPrice.multiply(num).setScale(2, BigDecimal.ROUND_HALF_UP);
            detail.set(OriginalBillConstant.Item.TAXAMOUNT, calAmount);
        } else if (!MathUtils.isNullOrZero(taxAmount) && !MathUtils.isNullOrZero(num) && !MathUtils.isNullOrZero(taxUnitPrice)) {
            LOGGER.info("金额，数量，单价都传输的情况...");
        } else if (!MathUtils.isNullOrZero(taxAmount) && !MathUtils.isNullOrZero(num) && MathUtils.isNullOrZero(taxUnitPrice)) {
            //传数量金额，未传单价
            try {
                BigDecimal price = UnitPriceHelper.calcPriceOrNum(taxAmount, num);
                detail.set(OriginalBillConstant.Item.TAXUNITPRICE, price);
            } catch (KDBizException e) {
                throw new KDBizException(String.format(ResManager.loadKDString("单据编号[%1$s]第[%2$s]行[%3$s]", "UpstreamToOriginalPlugin_8", "bos-ext-imc"), billNo, index, e.getMessage()));
            }
        } else if (!MathUtils.isNullOrZero(taxAmount) && MathUtils.isNullOrZero(num) && !MathUtils.isNullOrZero(taxUnitPrice)) {
            //传单价金额，未传数量
            try {
                BigDecimal quantity = UnitPriceHelper.calcPriceOrNum(taxAmount, taxUnitPrice);
                detail.set(OriginalBillConstant.Item.NUM, quantity);
            } catch (KDBizException e) {
                throw new KDBizException(String.format(ResManager.loadKDString("单据编号[%1$s]第[%2$s]行[%3$s]", "UpstreamToOriginalPlugin_8", "bos-ext-imc"), billNo, index, e.getMessage()));
            }
        } else{
            LOGGER.info(String.format("单据编号【%s】传输的明细行数据存在问题", SerializationUtils.toJsonString(detail)));
            throw new KDBizException(String.format(String.format(ResManager.loadKDString("单据编号[%1$s]第[%2$s]行，含税情况下，含税单价、含税金额、数量不能都不传输",
                    "UpstreamToOriginalPlugin_10", "bos-ext-imc"), billNo, index)));
        }

        BigDecimal taxRate = new BigDecimal(detail.getString(OriginalBillConstant.Item.TAXRATE));
        BigDecimal unitPrice = detail.getBigDecimal(OriginalBillConstant.Item.UNITPRICE);
        BigDecimal tax = detail.getBigDecimal(OriginalBillConstant.Item.TAX);
        BigDecimal amount = detail.getBigDecimal(OriginalBillConstant.Item.AMOUNT);
        //计算不含税单价
        if(MathUtils.isNullOrZero(unitPrice)
                || (taxUnitPrice!=null && unitPrice.compareTo(taxUnitPrice)==0)){
            BigDecimal excludingPrice = UnitPriceHelper.getUnitPriceByHS(taxRate, detail.getBigDecimal(OriginalBillConstant.Item.NUM),
                    null, detail.getBigDecimal(OriginalBillConstant.Item.TAXUNITPRICE));
            detail.set(OriginalBillConstant.Item.UNITPRICE, excludingPrice);
        }
        //计算不含税金额
        if(MathUtils.isNullOrZero(amount) || (taxAmount!=null && amount.compareTo(taxAmount)==0)) {
            BigDecimal excludingAmount = detail.getBigDecimal(OriginalBillConstant.Item.UNITPRICE).
                    multiply(detail.getBigDecimal(OriginalBillConstant.Item.NUM)).setScale(2, BigDecimal.ROUND_HALF_UP);
            detail.set(OriginalBillConstant.Item.AMOUNT, excludingAmount);
        }
        //计算税额
        if(MathUtils.isNullOrZero(tax)){
            BigDecimal excludingTax = detail.getBigDecimal(OriginalBillConstant.Item.TAXAMOUNT).
                    subtract(detail.getBigDecimal(OriginalBillConstant.Item.AMOUNT));
            detail.set(OriginalBillConstant.Item.TAX, excludingTax);
        }

        detail.set(OriginalBillConstant.Item.REMAINVALIDTAX, detail.getBigDecimal(OriginalBillConstant.Item.TAX));
        //含税使用含税金额
        detail.set(OriginalBillConstant.Item.REMAINVALIDAMOUNT,
                detail.getBigDecimal(OriginalBillConstant.Item.AMOUNT).add(detail.getBigDecimal(OriginalBillConstant.Item.TAX)));
        detail.set(OriginalBillConstant.Item.REMAINVALIDNUM, detail.getBigDecimal(OriginalBillConstant.Item.NUM));
    }

    /**
     * 设置金额保留两位
     * @param billItem
     */
    private static void setItemAmountInfo(DynamicObject billItem){
        //单据传入后金额，税额，折扣金额强制两位小数
        if (!MathUtils.isNullOrZero(billItem.getBigDecimal(OriginalBillConstant.Item.AMOUNT))) {
            billItem.set(OriginalBillConstant.Item.AMOUNT,
                    billItem.getBigDecimal(OriginalBillConstant.Item.AMOUNT).setScale(2, BigDecimal.ROUND_HALF_UP));
        }
        if (!MathUtils.isNullOrZero(billItem.getBigDecimal(OriginalBillConstant.Item.TAXAMOUNT))) {
            billItem.set(OriginalBillConstant.Item.TAXAMOUNT,
                    billItem.getBigDecimal(OriginalBillConstant.Item.TAXAMOUNT).setScale(2, BigDecimal.ROUND_HALF_UP));
        }
        if (!MathUtils.isNullOrZero(billItem.getBigDecimal(OriginalBillConstant.Item.TAX))) {
            billItem.set(OriginalBillConstant.Item.TAX,
                    billItem.getBigDecimal(OriginalBillConstant.Item.TAX).setScale(2, BigDecimal.ROUND_HALF_UP));
        }
        if (!MathUtils.isNullOrZero(billItem.getBigDecimal(OriginalBillConstant.Item.DISCOUNTAMOUNT))) {
            billItem.set(OriginalBillConstant.Item.DISCOUNTAMOUNT,
                    billItem.getBigDecimal(OriginalBillConstant.Item.DISCOUNTAMOUNT).setScale(2, BigDecimal.ROUND_HALF_UP));
        }
    }

    /**
     * 匹配税率信息
     * @param initTaxCodeMap
     * @param goodsCode
     * @return
     */
    private static String setDetailItemByGoodsCode(Map<String, DynamicObject> initTaxCodeMap, String goodsCode){
        DynamicObject taxCodeObj = initTaxCodeMap.get(goodsCode);
        if(taxCodeObj == null){
            return null;
        }
        BigDecimal taxRate = taxCodeObj.getBigDecimal(TaxClassCode.TAX_RATE);
        if(taxRate == null){
            return null;
        }

        String taxRateStr = TaxRateUtil.convertTaxRate(taxRate.stripTrailingZeros().toPlainString()); //税率 （小数 如17%即为0.17）
        return taxRateStr;
    }


    /**
     * 初始化税收分类编码数据
     * @param bills
     * @return
     */
    private static Map<String, DynamicObject> initTaxCodeMap(List<DynamicObject> bills){
        Map<String, DynamicObject> taxCodeMap = new HashMap<>(8);
        Set<String> taxCodeSet = new HashSet<>();
        bills.forEach((DynamicObject bill)->{
            bill.getDynamicObjectCollection(OriginalBillConstant.ITEM_KEY).forEach((DynamicObject itemObj)->{
                String goodsCode = itemObj.getString(OriginalBillConstant.Item.GOODSCODE);
                if(StringUtils.isNotEmpty(goodsCode)){
                    taxCodeSet.add(goodsCode);
                }
            });
        });

        //税收分类编码
        if(taxCodeSet.isEmpty()){
            return taxCodeMap;
        }
        List<String> codeList = taxCodeSet.stream().collect(Collectors.toList());
        DynamicObject[] dataList = getTaxClassByMergeCode(codeList);
        if(dataList==null || dataList.length==0){
            return taxCodeMap;
        }

        for(DynamicObject codeObj : dataList){
            taxCodeMap.put(codeObj.getString(TaxClassCode.MERGE_CODE), codeObj);
        }
        return taxCodeMap;
    }

    /**
     * 通过税收分类编码获取分类编码信息
     * @param mergeCodeList
     * @return
     */
    public static DynamicObject[] getTaxClassByMergeCode(List<String> mergeCodeList){
        QFilter taxFilter = new QFilter(TaxClassCode.HZX, QFilter.equals, TaxClassCode.HzxValue.NOT_HZX)
                .and(TaxClassCode.MERGE_CODE, QCP.in, mergeCodeList)
                .and(TaxClassCode.ENABLE, QCP.equals, CommonConstant.YES);
        return BusinessDataServiceHelper.load(TaxClassCode.FORM_ID,
                PropertyUtils.getAllPropertiesSplitByComma(TaxClassCode.FORM_ID, false), taxFilter.toArray());
    }
}
