/*
 * Decompiled with CFR 0.152.
 */
package kd.bos.gptas.adapter.vectordb.milvus.service;

import io.milvus.grpc.QueryResults;
import io.milvus.grpc.SearchResults;
import io.milvus.param.collection.DropCollectionParam;
import io.milvus.param.dml.DeleteParam;
import io.milvus.param.dml.InsertParam;
import io.milvus.param.dml.QueryParam;
import io.milvus.response.QueryResultsWrapper;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;
import kd.bos.context.RequestContext;
import kd.bos.gptas.adapter.vectordb.milvus.client.MilvusCollectionManager;
import kd.bos.gptas.adapter.vectordb.milvus.config.MilvusConfig;
import kd.bos.gptas.adapter.vectordb.milvus.service.MilvusConverter;
import kd.bos.gptas.api.vector.VectorResult;
import kd.bos.gptas.common.GptasErrorCode;
import kd.bos.gptas.common.vectordb.exception.VectorException;
import kd.bos.gptas.common.vectordb.model.VectorChunk;
import kd.bos.gptas.common.vectordb.model.VectorQuery;
import kd.bos.gptas.common.vectordb.service.KnowledgeVectorStoreService;
import kd.bos.logging.Log;
import kd.bos.logging.LogFactory;
import kd.bos.util.Pair;

public class MilvusVectorStore
extends KnowledgeVectorStoreService {
    private static final Log logger = LogFactory.getLog(MilvusVectorStore.class);
    private static final int BATCH_SIZE = 1000;
    private final AtomicBoolean initialized = new AtomicBoolean();
    private final MilvusCollectionManager collectionManager;
    private final MilvusConverter converter;
    private final MilvusConfig config;
    private final String formId;
    private boolean newVersionCollection;

    public MilvusVectorStore(MilvusConfig config, String formId) {
        this.config = config;
        this.collectionManager = new MilvusCollectionManager(config);
        this.formId = formId;
        this.converter = new MilvusConverter(formId);
    }

    public synchronized void initialize() throws VectorException {
        if (this.initialized.compareAndSet(false, true)) {
            try {
                String collectionName = this.generateCollectionName();
                logger.info("Initializing Milvus collection: {}", (Object)collectionName);
                this.collectionManager.initialize(collectionName);
                this.newVersionCollection = this.collectionManager.isNewVersionCollection(collectionName);
                logger.info("Successfully initialized Milvus collection: {}", (Object)collectionName);
            }
            catch (VectorException e) {
                this.initialized.set(false);
                throw e;
            }
        }
    }

    public List<VectorResult> doSearch(VectorQuery query) throws VectorException {
        this.validateQuery(query);
        Long l = System.currentTimeMillis();
        String collectionName = this.generateCollectionName();
        logger.debug("Performing vector search in collection: {}, topK: {}", (Object)collectionName, (Object)Objects.toString(query.getTopK(), "NULL"));
        try {
            SearchResults results = this.collectionManager.search(collectionName, query.getQueryVector(), query.getChunkTypes(), query.getKnowledgeIds(), query.getChunkIds(), this.newVersionCollection ? query.getGroupIds() : null, query.getTopK());
            List<VectorResult> vectorResults = this.converter.toVectorResults(results);
            List<VectorResult> filterVectorResults = this.filterResults(vectorResults, query.getLimitScore());
            logger.info("Search completed. Results: {}, filtered: {}, cost:{} ms ", new Object[]{vectorResults.size(), filterVectorResults != null ? filterVectorResults.size() : 0, System.currentTimeMillis() - l});
            return filterVectorResults;
        }
        catch (Exception e) {
            logger.error("Search operation failed", (Throwable)e);
            throw new VectorException(GptasErrorCode.VECTOR_STORE_READ_ERROR, "Search operation failed: " + e.getMessage(), (Throwable)e);
        }
    }

    public boolean doStore(VectorChunk vectorChunk) throws VectorException {
        this.validateVector(vectorChunk);
        try {
            logger.debug("Storing single vector chunk with id: {}", vectorChunk.getChunk() != null ? Long.valueOf(vectorChunk.getChunk().getId()) : "NULL");
            this.doStore(Collections.singletonList(vectorChunk));
            logger.info("Successfully stored vector chunk: {}", vectorChunk.getChunk() != null ? Long.valueOf(vectorChunk.getChunk().getId()) : "NULL");
        }
        catch (Exception e) {
            logger.error("Failed to store vector chunk: " + vectorChunk.getChunk().getId(), (Throwable)e);
            throw new VectorException(GptasErrorCode.VECTOR_STORE_WRITE_ERROR, "Store operation failed: " + e.getMessage(), (Throwable)e);
        }
        return true;
    }

    public Pair<List<VectorChunk>, List<VectorChunk>> doBatchStore(List<VectorChunk> vectorChunks) throws VectorException {
        if (vectorChunks == null || vectorChunks.isEmpty()) {
            return new Pair(new ArrayList(1), new ArrayList(1));
        }
        logger.info("Starting batch store operation for {} chunks", (Object)vectorChunks.size());
        for (VectorChunk vectorChunk : vectorChunks) {
            this.validateVector(vectorChunk);
        }
        ArrayList<VectorChunk> failedChunks = new ArrayList<VectorChunk>();
        ArrayList<VectorChunk> sucessedChunks = new ArrayList<VectorChunk>();
        List<List<VectorChunk>> batches = this.partition(vectorChunks, 1000);
        logger.info("Split into {} batches of size {}", (Object)batches.size(), (Object)1000);
        for (List<VectorChunk> batch : batches) {
            try {
                this.doStore(batch);
                sucessedChunks.addAll(batch);
                logger.info("Successfully stored batch of {} chunks", (Object)batch.size());
            }
            catch (Exception e) {
                logger.error("Batch store failed for " + batch.size() + " chunks", (Throwable)e);
                failedChunks.addAll(batch);
            }
        }
        if (!failedChunks.isEmpty()) {
            logger.warn("Failed to store {} chunks out of {}", (Object)failedChunks.size(), (Object)vectorChunks.size());
        }
        return new Pair(sucessedChunks, failedChunks);
    }

    public void delete(String entityId, List<Long> chunkIds) throws VectorException {
        if (chunkIds == null || chunkIds.isEmpty()) {
            return;
        }
        try {
            String collectionName = this.generateCollectionName();
            logger.debug("Deleting {} chunks from collection: {}", (Object)chunkIds.size(), (Object)collectionName);
            if (!this.collectionManager.existsCollection(collectionName)) {
                logger.info("Collection {} does not exist, skipping delete operation", (Object)collectionName);
                return;
            }
            String expr = this.buildDeleteExpression(chunkIds);
            DeleteParam deleteParam = DeleteParam.newBuilder().withCollectionName(collectionName).withExpr(expr).build();
            this.collectionManager.delete(deleteParam);
            logger.info("Successfully deleted {} chunks from collection: {}", (Object)chunkIds.size(), (Object)collectionName);
        }
        catch (Exception e) {
            logger.error("Failed to delete chunks: " + chunkIds, (Throwable)e);
            throw new VectorException(GptasErrorCode.VECTOR_STORE_DELETE_ERROR, "Delete operation failed: " + e.getMessage(), (Throwable)e);
        }
    }

    public void close() {
        try {
            logger.info("Closing MilvusVectorStore client");
            this.collectionManager.close();
            logger.info("Successfully closed MilvusVectorStore client");
        }
        catch (Exception e) {
            logger.error("Error closing Milvus client", (Throwable)e);
        }
    }

    private void validateVector(VectorChunk vectorChunk) {
        if (vectorChunk.getVector().size() != this.config.getDimension()) {
            throw new IllegalArgumentException("Invalid vector dimension: expected " + this.config.getDimension() + ", got " + vectorChunk.getVector().size());
        }
    }

    private void validateQuery(VectorQuery query) {
        if (query == null) {
            throw new IllegalArgumentException("Query cannot be null");
        }
        if (query.getQueryVector() == null && query.getQueryText() == null) {
            throw new IllegalArgumentException("Either query vector or query text must be provided");
        }
        if (query.getTopK() <= 0 || query.getTopK() > 1000) {
            throw new IllegalArgumentException("Invalid topK value: " + query.getTopK());
        }
    }

    private void doStore(List<VectorChunk> vectorChunks) throws VectorException {
        String collectionName = this.generateCollectionName();
        logger.info("Storing {} vectors in collection: {}", (Object)vectorChunks.size(), (Object)collectionName);
        List<InsertParam.Field> fields = this.converter.toMilvusFields(vectorChunks, this.newVersionCollection);
        InsertParam insertParam = InsertParam.newBuilder().withCollectionName(this.generateCollectionName()).withFields(fields).build();
        this.collectionManager.insert(insertParam);
    }

    private List<VectorResult> filterResults(List<VectorResult> results, float limitScore) {
        return results.stream().filter(r -> r.getScore() <= limitScore).collect(Collectors.toList());
    }

    private String buildDeleteExpression(List<Long> chunkIds) {
        return "id in [" + chunkIds.stream().map(String::valueOf).collect(Collectors.joining(",")) + "]";
    }

    private List<List<VectorChunk>> partition(List<VectorChunk> chunks, int batchSize) {
        ArrayList<List<VectorChunk>> batches = new ArrayList<List<VectorChunk>>(chunks.size());
        for (int i = 0; i < chunks.size(); i += batchSize) {
            batches.add(chunks.subList(i, Math.min(chunks.size(), i + batchSize)));
        }
        return batches;
    }

    public Map<String, List<?>> query(String entityId, List<Long> knowledgeIds, long start, long limit) {
        QueryParam.Builder builder = QueryParam.newBuilder().withCollectionName(this.generateCollectionName()).withOutFields(Arrays.asList("id", "vector", "knowledgeId")).withOffset(Long.valueOf(start)).withLimit(Long.valueOf(limit));
        if (knowledgeIds != null && !knowledgeIds.isEmpty()) {
            String knowledgeIdsExpr = " knowledgeId in [" + knowledgeIds.stream().map(String::valueOf).collect(Collectors.joining(",")) + "]";
            builder.withExpr(knowledgeIdsExpr);
        } else {
            builder.withExpr("knowledgeId > 0");
        }
        QueryParam queryParam = builder.build();
        QueryResults queryResults = this.collectionManager.query(queryParam);
        QueryResultsWrapper queryResultsWrapper = new QueryResultsWrapper(queryResults);
        HashMap map = new HashMap(16);
        map.put("id", queryResultsWrapper.getFieldWrapper("id").getFieldData());
        map.put("vector", queryResultsWrapper.getFieldWrapper("vector").getFieldData());
        map.put("knowledgeId", queryResultsWrapper.getFieldWrapper("knowledgeId").getFieldData());
        return map;
    }

    private String generateCollectionName() {
        String accountId = RequestContext.get().getAccountId();
        return String.format("KM_%s_%s_%s", accountId, this.formId, this.config.getEmbeddingName());
    }

    public void deleteAll(String entityId) {
        String collectionName = this.generateCollectionName();
        DropCollectionParam dropCollectionParam = DropCollectionParam.newBuilder().withCollectionName(collectionName).build();
        this.collectionManager.dropCollection(dropCollectionParam);
    }

    @FunctionalInterface
    private static interface SupplierWithException<T> {
        public T get() throws Exception;
    }
}

