package kd.bos.ext.imc.common.util;
import kd.bos.dataentity.resource.ResManager;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import kd.bos.dataentity.entity.DynamicObject;
import kd.bos.dataentity.entity.DynamicObjectCollection;
import kd.bos.dataentity.metadata.IDataEntityProperty;
import kd.bos.dataentity.metadata.IDataEntityType;
import kd.bos.dataentity.metadata.clr.DataEntityPropertyCollection;
import kd.bos.dataentity.utils.StringUtils;
import kd.bos.entity.property.*;
import kd.bos.exception.KDBizException;
import kd.bos.ext.imc.common.annotation.BeanFieldAnnotation;
import kd.bos.ext.imc.common.constant.CommonConstant;
import kd.bos.form.control.Control;
import kd.bos.form.plugin.AbstractFormPlugin;
import kd.bos.logging.Log;
import kd.bos.logging.LogFactory;
import kd.bos.orm.util.CollectionUtils;
import kd.bos.orm.util.ReflectionUtils;
import kd.bos.servicehelper.BusinessDataServiceHelper;

import java.lang.reflect.Field;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.text.MessageFormat;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author : rd_wang_yi
 * @description :
 * @date : 2020年08月28日 14:15
 **/
public class DynamicObjectUtil {
    private static final Log log = LogFactory.getLog(DynamicObjectUtil.class);

    public static <T> T dynamicObject2Bean(Class<T> clazz, DynamicObject dynamicObject) {
        T entity = null;
        try {
            DataEntityPropertyCollection collect = dynamicObject.getDataEntityType().getProperties();

            entity = clazz.newInstance();
            List<Field> fields = getDeclaredFields(entity);

            for (Field field : fields) {
                try {
                    ReflectionUtils.makeAccessible(field);
                    BeanFieldAnnotation beanField = field.getAnnotation(BeanFieldAnnotation.class);
                    if (beanField != null) {
                        String name = beanField.dynamicFiled();
                        if (collect.containsKey(name)) {
                            setField(dynamicObject, beanField, field, entity);
                        }
                    }
                } catch (Exception e) {
                    log.info(MessageFormat.format("dynamicObject转换失败,实体名:[{0}],字段名:[{1}]", dynamicObject.getDataEntityType().getName(), field.getName()));
                }
            }
        } catch (Exception e) {
            log.info("dynamicObject转换失败,实体名:[" + dynamicObject.getDataEntityType().getName() + "]", e);
        }
        return entity;
    }


    private static <T> List<Field> getDeclaredFields(T bean) {
        List<Field> fields = new ArrayList<>(Arrays.asList(bean.getClass().getDeclaredFields()));
        if (bean.getClass().getClassLoader() != null) {
            fields.addAll(Arrays.asList(bean.getClass().getSuperclass().getDeclaredFields()));
        }
        return fields;
    }

    public static <T> DynamicObject bean2DynamicObject(T bean, DynamicObject dynamicObject) {
        DataEntityPropertyCollection collect = dynamicObject.getDataEntityType().getProperties();

        List<Field> fields = getDeclaredFields(bean);
        for (Field field : fields) {
            try {
                ReflectionUtils.makeAccessible(field);
                BeanFieldAnnotation beanField = field.getAnnotation(BeanFieldAnnotation.class);
                if (beanField != null) {
                    String name = beanField.dynamicFiled();
                    IDataEntityProperty property = collect.get(name);
                    if (property == null) {
                        continue;
                    }
                    Object value = getValue(beanField, field, dynamicObject, bean);
                    if (property.getPropertyType() != null && Date.class.getSimpleName().equals(property.getPropertyType().getSimpleName()) && value instanceof String && StringUtils.isNotBlank(value)) {
                        value = DateUtils.handleDate((String) value);
                    }
                    dynamicObject.set(name, value);
                }
            } catch (Exception e) {
                log.info(MessageFormat.format("bean转DynamicObject失败, 字段名:{0},bean类型:{1}", field.getName(), bean.getClass()));
            }
        }
        return dynamicObject;
    }

    @SuppressWarnings("unchecked")
    public static void map2DynamicObject(Map<String, Object> map, DynamicObject dynamicObject) {
        DataEntityPropertyCollection collect = dynamicObject.getDataEntityType().getProperties();
        for (IDataEntityProperty property : collect) {
            String field = property.getName();
            Object value = map.get(field);
            if (StringUtils.isBlank(value)) {
                dynamicObject.set(field, null);
                continue;
            }

            if (property instanceof EntryProp) {
                DynamicObjectCollection collection = dynamicObject.getDynamicObjectCollection(field);
                List<Map<String, Object>> list = (List<Map<String, Object>>) map.get(field);
                if (!CollectionUtils.isEmpty(list)) {
                    list.forEach(o -> map2DynamicObject(o, collection.addNew()));
                }
                continue;
            }

            if (property instanceof BasedataProp && value instanceof Map) {
                dynamicObject.set(field, ((Map) value).get(CommonConstant.ID));
                continue;
            }
            if (BigDecimal.class.getSimpleName().equals(property.getPropertyType().getSimpleName()) && StringUtils.isNotBlank(value)) {
                if(value instanceof String){
                    if (!BigDecimalUtil.isNumber((String) value)) {
                        dynamicObject.set(field, BigDecimal.ZERO);
                    } else {
                        dynamicObject.set(field, value);
                    }
                    continue;
                }
            }

            if ((property instanceof DateTimeProp) && StringUtils.isNotBlank(value)) {
                String date = String.valueOf(value);
                String dateFormat = DateUtils.YYYY_MM_DD;
                if(date.length() == 19){
                    dateFormat = DateUtils.YYYY_MM_DD_HH_MM_SS;
                }else if (date.length() == 8){
                    dateFormat = DateUtils.YYYYMMDD;
                }
                if (date.length() != 10 && date.length() != 19 && date.length() != 8) {
                    throw new KDBizException(String.format(ResManager.loadKDString("日期[%1$s]格式错误,正确格式为:%2$s", "DynamicObjectUtil_0", "bos-ext-imc") , date, DateUtils.YYYY_MM_DD));
                }
                dynamicObject.set(field, DateUtils.stringToDate(date, dateFormat));
                continue;
            }

            dynamicObject.set(field, value);
        }
    }

    public static Map<String, Object> dynamicObject2Map(DynamicObject dynamicObject) {
        DataEntityPropertyCollection collect = dynamicObject.getDataEntityType().getProperties();

        Map<String, Object> map = Maps.newHashMapWithExpectedSize(collect.size());
        for (IDataEntityProperty property : collect) {
            String field = property.getName();
            map.put(field, dynamicObject.get(field));
        }
        return map;
    }

    public static Map<String, Object> dynamicObjectToMap(DynamicObject obj, boolean id2String) {
        if (obj == null) {
            return null;
        }
        Map<String, Object> map = new HashMap<>(obj.getDynamicObjectType().getProperties().size());
        DataEntityPropertyCollection collect = obj.getDynamicObjectType().getProperties();
        for (IDataEntityProperty property : collect) {
            String field = property.getName();
            if (property instanceof EntryProp) {
                DynamicObjectCollection collection = obj.getDynamicObjectCollection(field);
                map.put(field, dynamicObjectCollection2ListMap(collection, id2String));
                continue;
            }

            Object fieldObj = obj.get(field);
            if (id2String) {
                if (property.getPropertyType() == Long.class || property.getPropertyType() == long.class || fieldObj instanceof Long) {
                    if (fieldObj != null) {
                        map.put(field, String.valueOf(fieldObj));
                    }
                    continue;
                }
            }
            if (fieldObj instanceof DynamicObject) {
                Object pkValue = ((DynamicObject) fieldObj).getPkValue();
                map.put(field, id2String ? pkValue.toString() : pkValue);
                continue;
            }
            if (fieldObj instanceof String) {
                String value = (String) fieldObj;
                map.put(field, htmlEncode(value));
                continue;
            }

            map.put(field, obj.get(field));
        }
        map.put("tableName", obj.getDataEntityType().getName());
        return map;
    }

    public static String htmlEncode(String source) {
        if (source == null) {
            return "";
        }
        StringBuilder buffer = new StringBuilder();
        for (int i = 0; i < source.length(); i++) {
            char c = source.charAt(i);
            switch (c) {
                case '<':
                    buffer.append('＜');
                    break;
                case '>':
                    buffer.append('＞');
                    break;
                case '&':
                    buffer.append('＆');
                    break;
                case '"':
                    buffer.append('“');
                    break;
                case '\\':
                    buffer.append('＼');
                    break;
                case '#':
                    buffer.append('＃');
                    break;
                default:
                    buffer.append(c);
            }
        }
        return buffer.toString();
    }

    public static Map<String, Object> dynamicObjectToMap(DynamicObject obj) {
        return dynamicObjectToMap(obj, false);
    }

    public static List<Map<String, Object>> dynamicObjectslist2ListMap(List<DynamicObject> objs) {
        List<Map<String, Object>> list = Lists.newArrayListWithCapacity(objs.size());
        for (DynamicObject obj : objs) {
            list.add(dynamicObjectToMap(obj, true));
        }
        return list;
    }

    public static List<Map<String, Object>> dynamicObjectCollection2ListMap(DynamicObjectCollection collection, boolean long2String) {
        List<Map<String, Object>> list = Lists.newArrayListWithCapacity(collection.size());
        for (DynamicObject obj : collection) {
            list.add(dynamicObjectToMap(obj, long2String));
        }
        return list;
    }

    public static List<Map<String, Object>> dynamicObjects2ListMap(List<DynamicObject> objs, boolean long2String) {
        List<Map<String, Object>> list = Lists.newArrayListWithCapacity(objs.size());
        for (DynamicObject obj : objs) {
            list.add(dynamicObjectToMap(obj, long2String));
        }
        return list;
    }

    public static List<Map<String, Object>> dynamicObjects2ListMap(DynamicObject[] objs, boolean long2String) {
        List<Map<String, Object>> list = Lists.newArrayListWithCapacity(objs.length);
        for (DynamicObject obj : objs) {
            list.add(dynamicObjectToMap(obj, long2String));
        }
        return list;
    }

    /**
     * @param source
     * @description 复制Dynamicobject，包含id
     * @author rd_wang_yi
     * @date 2021/1/20 16:38
     */
    public static DynamicObject copyDynamicObject(DynamicObject source) {
        DynamicObject newObj = (DynamicObject) source.getDataEntityType().createInstance();
        copyDynamicObject(source, newObj, true);
        return newObj;
    }

    /**
     * @param source
     * @description 复制Dynamicobject，不包含id
     * @author rd_wang_yi
     * @date 2021/1/20 16:38
     */
    public static void copyDynamicObject(DynamicObject source, DynamicObject target) {
        copyDynamicObject(source, target, false);
    }

    public static void copyDynamicObject(DynamicObject source, DynamicObject target, boolean containId) {
        copyDynamicObject(source, target, containId, true, false);
    }

    public static void copyDynamicObject(DynamicObject source, DynamicObject target, boolean containId, boolean nullOvrride, boolean copyBaseData) {
        DataEntityPropertyCollection sourceCollect = source.getDataEntityType().getProperties();

        DataEntityPropertyCollection collect = target.getDataEntityType().getProperties();
        for (IDataEntityProperty property : collect) {
            // 附件不复制
            if (property instanceof AttachmentProp) {
                continue;
            }
            if (!copyBaseData) {
                if (property instanceof BasedataProp || property instanceof CreateDateProp) {
                    continue;
                }
            }
            if (!containId && CommonConstant.ID.equals(property.getName())) {
                continue;
            }
            String field = property.getName();
            if (property instanceof EntryProp && sourceCollect.containsKey(field)) {
                DynamicObjectCollection collection = source.getDynamicObjectCollection(field);
                for (DynamicObject dynamicObject : collection) {
                    copyDynamicObject(dynamicObject, target.getDynamicObjectCollection(field).addNew(), containId, nullOvrride, copyBaseData);
                }
                continue;
            }
            if (sourceCollect.containsKey(field)) {
                Object value = source.get(field);
                if (!nullOvrride && checkValueIsNull(value)) {
                    continue;
                }
                target.set(field, value);
            }
        }
    }

    private static boolean checkValueIsNull(Object value) {
        if (value == null) {
            return true;
        }
        Class<?> clazz = value.getClass();
        if (clazz == BigDecimal.class && ((BigDecimal) value).compareTo(BigDecimal.ZERO) == 0) {
            return true;
        }
        if (clazz == Long.class && value.equals(0L)) {
            return true;
        }
        return StringUtils.isBlank(value);
    }

    /**
     * @Description: 复制 DynamicObject 到 View
     * @Author: jimmy_zhu
     * @Date: 2021/7/21 15:22
     **/
    public static void copyDynamicObject2View(DynamicObject source, AbstractFormPlugin formPlugin, boolean isItem, int index) {
        DataEntityPropertyCollection collect = source.getDataEntityType().getProperties();
        for (IDataEntityProperty property : collect) {
//            if (property instanceof BasedataProp || property instanceof CreateDateProp) {
//                continue;
//            }
            if (CommonConstant.ID.equals(property.getName())) {
                continue;
            }
            String field = property.getName();
            Control control = formPlugin.getControl(field);
            if (control != null) {
                if (isItem) {
                    formPlugin.getView().getModel().setValue(field, source.get(field), index);
                } else {
                    formPlugin.getView().getModel().setValue(field, source.get(field));
                }
            }
        }
    }

    public static void copyDynamicObjectWithOutItems(DynamicObject source, DynamicObject target) {
        DataEntityPropertyCollection sourceCollect = source.getDataEntityType().getProperties();
        DataEntityPropertyCollection collect = target.getDataEntityType().getProperties();
        for (IDataEntityProperty property : collect) {
            String field = property.getName();
            if (!sourceCollect.containsKey(field) || property instanceof CreateDateProp) {
                continue;
            }
            if (property instanceof BasedataProp) {
                target.set(field, DynamicObjectUtil.getDynamicObjectLongValue(source.get(field)));
            }
            if (!(property instanceof EntryProp)) {
                if (sourceCollect.containsKey(field)) {
                    target.set(field, source.get(field));
                }
            }
        }
    }

    private static <T> Object getValue(BeanFieldAnnotation beanField, Field field, DynamicObject dynamicObject, T entity) {
        try {
            Class<?> fieldType = field.getType();
            if (List.class == fieldType) {
                List list = (List) field.get(entity);
                DynamicObjectCollection collection = dynamicObject.getDynamicObjectCollection(beanField.dynamicFiled());
                if(list != null){
                    for (Object o : list) {
                        bean2DynamicObject(o, collection.addNew());
                    }
                }
                return collection;
            } else {
                if (null != fieldType.getClassLoader()) {
                    return bean2DynamicObject(field.get(entity), dynamicObject.getDynamicObject(beanField.dynamicFiled()));
                }
                return field.get(entity);
            }
        } catch (Exception e) {
            log.info("getValue失败, 字段名" + field.getName(), e);
        }
        return null;
    }

    @SuppressWarnings("unchecked")
    private static <T> void setField(DynamicObject dynamicObject, BeanFieldAnnotation beanField, Field field, T entity) throws IllegalAccessException {
        Class<?> fieldType = field.getType();
        if (String.class == fieldType) {
            if (beanField.classType() == BigDecimal.class) {
                field.set(entity, dynamicObject.getBigDecimal(beanField.dynamicFiled()).setScale(beanField.scale(), BigDecimal.ROUND_HALF_UP).stripTrailingZeros().toPlainString());
            } else if (dynamicObject.get(beanField.dynamicFiled()) instanceof DynamicObject) {
                field.set(entity, dynamicObject.getDynamicObject(beanField.dynamicFiled()).getPkValue().toString());
            } else {
                field.set(entity, dynamicObject.getString(beanField.dynamicFiled()));
            }
        } else if ((Integer.TYPE == fieldType) || (Integer.class == fieldType)) {
            field.set(entity, dynamicObject.getInt(beanField.dynamicFiled()));
        } else if ((Long.TYPE == fieldType) || (Long.class == fieldType)) {
            if (dynamicObject.get(beanField.dynamicFiled()) instanceof DynamicObject) {
                field.set(entity, dynamicObject.getDynamicObject(beanField.dynamicFiled()).getPkValue());
            } else {
                field.set(entity, dynamicObject.getLong(beanField.dynamicFiled()));
            }
        } else if ((BigDecimal.class == fieldType)) {
            field.set(entity, getBigDecimalVal(dynamicObject, beanField));
        } else if (Date.class == fieldType) {
            field.set(entity, dynamicObject.getDate(beanField.dynamicFiled()));
        } else if (LocalDateTime.class == fieldType) {
            Date date = dynamicObject.getDate(beanField.dynamicFiled());
            if(date != null){
                LocalDateTime localDateTime = LocalDateTime.ofInstant(date.toInstant(), ZoneId.systemDefault());
                field.set(entity, localDateTime);
            }
        } else if (BigInteger.class == fieldType) {
            field.set(entity, getBigIntegerVal(dynamicObject, beanField));
        } else if (List.class == fieldType) {
            DynamicObjectCollection collection = dynamicObject.getDynamicObjectCollection(beanField.dynamicFiled());
            List newList = new ArrayList();
            for (DynamicObject obj : collection) {
                newList.add(dynamicObject2Bean(beanField.classType(), obj));
            }
            field.set(entity, newList);
        } else {
            if (fieldType.getClassLoader() != null) {
                DynamicObject obj = dynamicObject.getDynamicObject(beanField.dynamicFiled());
                field.set(entity, dynamicObject2Bean(fieldType, obj));
            } else {
                field.set(entity, dynamicObject.get(beanField.dynamicFiled()));
            }
        }
    }

    private static BigInteger getBigIntegerVal(DynamicObject dynamicObject, BeanFieldAnnotation beanField) {
        BigInteger integer = BigInteger.ZERO;
        try {
            Object valObject = dynamicObject.get(beanField.dynamicFiled());
            if (valObject != null) {
                String val = String.valueOf(valObject);
                integer = new BigInteger(val);
            }
        } catch (Exception e) {
            log.info("BigInteger转换失败,字段名:" + beanField.dynamicFiled(), e);
        }
        return integer;
    }

    private static BigDecimal getBigDecimalVal(DynamicObject obj, BeanFieldAnnotation beanField) {
        BigDecimal val = obj.getBigDecimal(beanField.dynamicFiled());
        if (val != null) {
            if (val.compareTo(BigDecimal.ZERO) == 0) {
                val = new BigDecimal("0.00");
            } else {
                if(beanField.scale() != 2){
                    val = new BigDecimal(val.setScale(beanField.scale(), BigDecimal.ROUND_HALF_UP).stripTrailingZeros().toPlainString());
                }else{
                    val = val.setScale(beanField.scale(), BigDecimal.ROUND_HALF_UP);
                }
            }
        }
        return val;
    }

    /**
     * @param dynamicObject
     * @description DynamicObject转json，不包含明细
     * @author rd_wang_yi
     * @date 2022/4/8 9:24
     */
    public static JSONObject dynamicObject2Json(DynamicObject dynamicObject) {
        JSONObject jsonObject = new JSONObject();
        IDataEntityType entityType = dynamicObject.getDataEntityType();
        DataEntityPropertyCollection collect = entityType.getProperties();
        for (IDataEntityProperty property : collect) {
            String fieldName = property.getName();
            if (dynamicObject.get(fieldName) != null) {
                if (!(collect.get(property.getName()) instanceof EntryProp)) {
                    Object value = dynamicObject.get(fieldName);
                    jsonObject.put(fieldName, value);
                }
            }
        }
        return jsonObject;
    }

    public static JSONObject dynamicObject2JsonContainItems(DynamicObject dynamicObject) {
        JSONObject json = new JSONObject();
        dynamicObject2JsonContainItems(dynamicObject, json);
        return json;
    }

    public static JSONObject dynamicObject2JsonContainItems(DynamicObject dynamicObject, JSONObject jsonObject) {
        IDataEntityType entityType = dynamicObject.getDataEntityType();
        DataEntityPropertyCollection collect = entityType.getProperties();
        for (IDataEntityProperty property : collect) {
            String fieldName = property.getName();
            if (dynamicObject.get(fieldName) != null) {
                if (!(collect.get(property.getName()) instanceof EntryProp)) {
                    Object value = dynamicObject.get(fieldName);
                    jsonObject.put(fieldName, value);
                    continue;
                }

                //将items部分也转成json
                DynamicObjectCollection items = dynamicObject.getDynamicObjectCollection(fieldName);
                JSONArray itemArray = new JSONArray();
                for (DynamicObject item : items) {
                    JSONObject itemJson = new JSONObject();
                    dynamicObject2JsonContainItems(item, itemJson);
                    itemArray.add(itemJson);
//                    IDataEntityType itemEntityType = item.getDataEntityType();
//                    DataEntityPropertyCollection itemCollect = itemEntityType.getProperties();
//                    JSONObject itemJson = new JSONObject();
//                    for (IDataEntityProperty itemProperty : itemCollect) {
//                        String itemfieldName = itemProperty.getName();
//                        if (item.get(itemfieldName) != null) {
//                            Object itemValue = item.get(itemfieldName);
//                            itemJson.put(itemfieldName, itemValue);
//                        }
//                    }
//                    itemArray.add(itemJson);
                }
                jsonObject.put(fieldName, itemArray);
            }
        }
        return jsonObject;
    }


    /**
     * @Description: json 转为 DynamicObject
     * @Author: jimmy_zhu
     * @Date: 2020/11/12 14:26
     **/
    public static DynamicObject json2DynamicObject(DynamicObject dynamicObject, JSONObject jsonObject) {
        IDataEntityType entityType = dynamicObject.getDataEntityType();
        DataEntityPropertyCollection collect = entityType.getProperties();
        for (IDataEntityProperty property : collect) {
            String fieldName = property.getName();
            if (null != jsonObject.get(fieldName) && !kd.bos.util.StringUtils.isEmpty(String.valueOf(jsonObject.get(fieldName)))) {
                if ("id".equals(fieldName)) {
                    continue;
                }
                if (!"items".equals(fieldName)) {
                    dynamicObject.set(fieldName, jsonObject.get(fieldName));
                }
            }
        }
        return dynamicObject;
    }

    public static long getDynamicObjectLongValue(Object value) {
        try {
            if (value instanceof DynamicObject) {
                return Long.parseLong(((DynamicObject) value).getPkValue().toString());
            } else {
                return Long.parseLong(String.valueOf(value));
            }
        } catch (Exception e) {
            return 0L;
        }
    }

    public static void json2DynamicObjectContainItems(JSONObject json, DynamicObject dynamicObject, boolean isContainId) {
        DataEntityPropertyCollection collect = dynamicObject.getDataEntityType().getProperties();
        Set<String> fields = collect.stream().map(IDataEntityProperty::getName).collect(Collectors.toSet());

        for (Map.Entry<String, Object> entry : json.entrySet()) {
            try {
                if (!isContainId && CommonConstant.ID.equals(entry.getKey())) {
                    continue;
                }
                if (fields.contains(entry.getKey())) {
                    if ("null".equals(entry.getValue())) {
                        dynamicObject.set(entry.getKey(), null);
                        continue;
                    }
                    Class clazz = collect.get(entry.getKey()).getPropertyType();
                    if (clazz == String.class) {
                        dynamicObject.set(entry.getKey(), entry.getValue());
                        continue;
                    }

                    if (clazz == Date.class) {
                        if (StringUtils.isNotBlank(entry.getValue())) {
                            String value = (String) entry.getValue();
                            dynamicObject.set(entry.getKey(), DateUtils.stringToDate(value, value.length() == 10 ? DateUtils.YYYY_MM_DD : DateUtils.YYYY_MM_DD_HH_MM_SS));
                            continue;
                        }
                    }

                    if (clazz == BigDecimal.class && StringUtils.isBlank(entry.getValue())) {
                        dynamicObject.set(entry.getKey(), BigDecimal.ZERO);
                        continue;
                    }

                    if (clazz == Integer.class && StringUtils.isBlank(entry.getValue())) {
                        dynamicObject.set(entry.getKey(), 0);
                        continue;
                    }

                    if (clazz == Long.class && StringUtils.isBlank(entry.getValue())) {
                        dynamicObject.set(entry.getKey(), 0L);
                        continue;
                    }

                    if (collect.get(entry.getKey()) instanceof BasedataProp && entry.getValue() instanceof JSONObject) {
                        JSONObject baseDataJson = (JSONObject) entry.getValue();
                        DynamicObject object;
                        String entityId = ((BasedataProp) collect.get(entry.getKey())).getBaseEntityId();
                        try {
                            object = BusinessDataServiceHelper.newDynamicObject(entityId);
                            json2DynamicObjectContainItems(baseDataJson, object, true);
                        } catch (Exception e) {
                            object = BusinessDataServiceHelper.loadSingleFromCache(baseDataJson.get(CommonConstant.ID), entityId);
                        }
                        dynamicObject.set(entry.getKey(), object);
                        continue;
                    }

                    if (collect.get(entry.getKey()) instanceof EntryProp) {
                        JSONArray jsonArray = (JSONArray) entry.getValue();
                        for (Object object : jsonArray) {
                            DynamicObject obj = dynamicObject.getDynamicObjectCollection(entry.getKey()).addNew();
                            json2DynamicObjectContainItems((JSONObject) object, obj, isContainId);
                        }
                        continue;
                    }
                    dynamicObject.set(entry.getKey(), entry.getValue());
                }
            } catch (Exception e) {
                log.error(MessageFormat.format("json转换DynamicObject失败{0}, key为{1}, 值为{2}", e.getMessage(), entry.getKey(), entry.getValue()), e);
            }
        }
    }

    public static void json2DynamicObjectContainItems(JSONObject json, DynamicObject dynamicObject) {
        json2DynamicObjectContainItems(json, dynamicObject, false);
    }


    /*
     * @Description 集合转数组
     * @date 2021/10/19 11:15
     * @param dynamicObjectList
     * @return
     */
//    public static DynamicObject[] DynamicObjectListConvertArr(List<DynamicObject> dynamicObjectList) {
//        DynamicObject[] billArr = new DynamicObject[dynamicObjectList.size()];
//        for (int i = 0; i < dynamicObjectList.size(); i++) {
//            billArr[i] = dynamicObjectList.get(i);
//        }
//        return billArr;
//    }

    /**
     * @param jsonArray
     * @param entityName
     * @return
     * @Description 将jsonArray转为List<DynamicObject>
     * @Author luchun_pi
     */
    public static List<DynamicObject> array2DynamicListContainItems(JSONArray jsonArray, String entityName) {
        List<DynamicObject> dynamicObjectList = Lists.newArrayListWithCapacity(4);
        if (jsonArray != null && jsonArray.size() > 0) {
            for (int i = 0; i < jsonArray.size(); i++) {
                DynamicObject dynamicObject = BusinessDataServiceHelper.newDynamicObject(entityName);
                JSONObject jsonObject = jsonArray.getJSONObject(i);
                json2DynamicObjectContainItems(jsonObject, dynamicObject);
                dynamicObjectList.add(dynamicObject);
            }
        }
        return dynamicObjectList;
    }

    /***
     * @Description: 检查某个DynamicObject是否含有某字段
     * @param: dynamicObject
     * @param: field
     * @return: boolean
     * @Author: jimmy_zhu
     * @Date: 2022/7/11 16:12
     */
    public static boolean checkDynamicObjectHasField(DynamicObject dynamicObject, String field) {
        IDataEntityType entityType = dynamicObject.getDataEntityType();
        DataEntityPropertyCollection collect = entityType.getProperties();
        for (IDataEntityProperty property : collect) {
            if (property.getName().equals(field)) {
                return true;
            }
        }
        return false;
    }
}
