package kd.bos.ext.imc.common.util;
import kd.bos.dataentity.resource.ResManager;

import kd.bos.dataentity.utils.StringUtils;
import kd.bos.exception.KDBizException;

import java.math.BigDecimal;
import java.math.RoundingMode;

/**
 * @author jimmy_zhu
 * @Description: 单价反算工具类
 * @date 2022/3/21 18:13
 */
public class UnitPriceHelper {

    public static BigDecimal getUnitPriceByHS(BigDecimal taxRate, BigDecimal num, BigDecimal amount, BigDecimal taxUnitPrice) {
        if (MathUtils.isZero(taxUnitPrice) || !MathUtils.isZero(amount)) {
            if (MathUtils.isZero(num)) {
                return BigDecimal.ZERO;
            }
            // 不含税单价 = 不含税金额 / 数量

            return calcPriceOrNum(amount,num);
        } else {
            // 不含税单价 = 含税单价/(1+税率)
            return getValue(taxUnitPrice.divide(BigDecimal.ONE.add(taxRate), 15, RoundingMode.HALF_UP),ResManager.loadKDString("不含税单价", "UnitPriceHelper_2", "bos-ext-imc"));
        }
    }

    public static BigDecimal getTaxUnitPriceByBHS(BigDecimal taxRate, BigDecimal num, BigDecimal taxAmount, BigDecimal unitPrice) {
        if (MathUtils.isZero(unitPrice) || !MathUtils.isZero(taxAmount)) {
            if (MathUtils.isZero(num)) {
                return BigDecimal.ZERO;
            }
            // 含税单价 = 含税金额 / 数量

            return calcPriceOrNum(taxAmount,num);
        } else {
            // 含税单价 = 不含税单价*(1+税率)
            return getValue(unitPrice.multiply(BigDecimal.ONE.add(taxRate)),ResManager.loadKDString("含税单价", "UnitPriceHelper_3", "bos-ext-imc"));
        }
    }

    public static BigDecimal calcPriceOrNum(BigDecimal divisor, BigDecimal dividend) {
        if (MathUtils.isNullOrZero(dividend)) {
            return BigDecimal.ZERO;
        }
        return divisor.divide(dividend, getMaxPointLength(divisor, dividend), BigDecimal.ROUND_HALF_UP).stripTrailingZeros();
    }

    /**
     * 通过不含税单价计算含税单价
     * @param unitPrice
     * @param taxRate
     * @return
     */
    public static BigDecimal calTaxUnitPrice(BigDecimal unitPrice, BigDecimal taxRate) {
        if (MathUtils.isNullOrZero(unitPrice) || MathUtils.isNullOrZero(taxRate)) {
            return BigDecimal.ZERO;
        }

        return unitPrice.multiply(BigDecimal.ONE.add(taxRate)).setScale(8, BigDecimal.ROUND_HALF_UP);
    }


    public static int getMaxPointLength(BigDecimal divisor, BigDecimal dividend) {
        int intLength = divisor.divide(dividend, 0, BigDecimal.ROUND_DOWN).abs().toPlainString().length();
        if(intLength > 13){
            throw new KDBizException(ResManager.loadKDString("计算得出的单价/数量整数位超长， 整数位最多13位", "UnitPriceHelper_0", "bos-ext-imc"));
        }
        int pointLength = 15 - intLength;
        if (pointLength > 13) {
            pointLength = 13;
        }
        return pointLength;
    }

    /**
     *
     * @Description 数据库的字段标准：（26,13），即整数位和小数位最多各13位，同时，整数+小数位+小数点最多16位
     * @Author  wenhao_wei
     * @Date   2023/3/28 10:30
     * @param: inputValue
     * @Return java.math.BigDecimal
     * @Exception 单价【3.3333333333】整数位超长
     *
     */
    public static BigDecimal getValue(BigDecimal inputValue,String name){
        if(null ==inputValue){
            return BigDecimal.ZERO;
        }
        String inputValueStr = inputValue.toPlainString();
        String replacedStr = inputValueStr.replace("-", "");

        if(!inputValueStr.contains(".") && replacedStr.length() > 13){
            throw new KDBizException(String.format(ResManager.loadKDString("%1$s[%2$s]整数位超长,整数位最多13位", "UnitPriceHelper_1", "bos-ext-imc"),name,inputValue));
        }
        String[] array = replacedStr.split("\\.");
        if (array[0].length() > 13) {
            throw new KDBizException(String.format(ResManager.loadKDString("%1$s[%2$s]整数位超长,整数位最多13位", "UnitPriceHelper_1", "bos-ext-imc"),name,inputValue));
        }
        if(array.length > 1){
            int firstLength = array[0].length();
            return inputValue.setScale(15 - firstLength, RoundingMode.HALF_UP).stripTrailingZeros();
        }else{
            return inputValue.stripTrailingZeros();
        }
    }
    public static String getStrValue(String inputValue,String name){
        if(StringUtils.isBlank(inputValue)){
            return inputValue;
        }
        if(!BigDecimalUtil.isNumber(inputValue)){
            return "";
        }
        return getValue(new BigDecimal(inputValue),name).toPlainString();
    }

    public static BigDecimal getValue(BigDecimal inputValue){
        return getValue(inputValue,"");
    }
}
